# Plans DataTable - TailAdmin Integration Guide

## Overview

A professional admin panel datatable for managing subscription plans, built with TailAdmin design system and Tailwind CSS. The interface is accessible at `/admin/plans`.

## Features Implemented

### ✅ Modern DataTable Design
- **Professional Layout**: Clean, modern card-based table design following TailAdmin standards
- **Responsive Design**: Fully responsive on mobile, tablet, and desktop
- **Dark Mode Support**: Complete dark mode theme with smooth transitions
- **Breadcrumb Navigation**: Easy navigation with breadcrumb at the top

### ✅ Search Functionality
- **Real-time Search**: Instant client-side search across all table data
- **Search Input**: Beautiful search bar with icon in the table header
- **Smart Filtering**: Searches across all plan fields (name, price, duration, etc.)

### ✅ Plan Management
- **Create Plans**: Modal form to add new subscription plans
- **Edit Plans**: Edit existing plans with pre-filled data
- **Delete Plans**: Confirmation modal before deletion
- **View Plans**: Beautiful table view with all plan details

### ✅ Visual Elements
- **Status Badges**: Color-coded badges for active/inactive status
- **Duration Pills**: Visual pills for monthly/yearly durations
- **Action Icons**: SVG icons for edit and delete actions
- **Empty State**: Friendly empty state when no plans exist

### ✅ Form Validation
- **Required Fields**: All essential fields are marked as required
- **Input Types**: Proper input types (number, text, select, textarea)
- **Placeholders**: Helpful placeholders in all form fields
- **Client & Server Validation**: Both frontend and backend validation

### ✅ TailAdmin Color Palette
Full TailAdmin color system integrated:
- Primary: `#3C50E0` (Blue)
- Success: `#10B981` (Green)
- Danger: `#F87171` (Red)
- Warning: `#FFC107` (Amber)
- Custom strokes, backgrounds, and text colors

## File Structure

```
resources/
├── views/
│   └── admin/
│       └── plans/
│           └── index.blade.php      # Main plans datatable page
├── css/
│   └── app.css                      # Custom styles
└── js/
    └── app.js                       # Main JavaScript

app/
├── Http/
│   └── Controllers/
│       └── Admin/
│           └── PlanController.php   # Plans CRUD controller
└── Models/
    └── SubscriptionPlan.php         # Plan model

routes/
└── web.php                          # Route definitions

tailwind.config.js                   # TailAdmin Tailwind configuration
```

## Route Information

### URL
```
/admin/plans
```

### Route Name
```php
route('plans.index')
```

### Available Routes
```php
Route::resource('plans', PlanController::class);
// This creates:
// GET    /admin/plans           - List all plans
// GET    /admin/plans/create    - Show create form (not used - modal)
// POST   /admin/plans           - Store new plan
// GET    /admin/plans/{id}      - Show single plan (JSON for AJAX)
// GET    /admin/plans/{id}/edit - Show edit form (not used - modal)
// PUT    /admin/plans/{id}      - Update plan
// DELETE /admin/plans/{id}      - Delete plan
```

## Usage Guide

### Creating a Plan

1. Click the **"Add Plan"** button in the top-right
2. Fill in the form fields:
   - **Plan Name** (required): e.g., "Basic", "Pro", "Enterprise"
   - **Price** (required): In INR (₹), e.g., 999.00
   - **Duration** (required): Monthly or Yearly
   - **Max Posts / Month** (required): Number of posts allowed
   - **Max Social Accounts** (required): Number of social accounts allowed
   - **AI Credits** (required): AI credits included
   - **Features** (optional): One feature per line
   - **Status** (required): Active or Inactive
3. Click **"Save Plan"**

### Editing a Plan

1. Click the **Edit icon** (pencil) in the Actions column
2. Modal opens with pre-filled data
3. Modify any fields you want to change
4. Click **"Save Plan"**

### Deleting a Plan

1. Click the **Delete icon** (trash) in the Actions column
2. Confirmation modal appears
3. Click **"Delete Plan"** to confirm
4. Plan is permanently deleted

### Searching Plans

1. Type in the search box at the top-right
2. Results filter in real-time
3. Search works across all visible fields

## Design Components

### Table Columns

| Column | Description | Visual Style |
|--------|-------------|--------------|
| ID | Plan ID number | `#` prefix |
| Plan Name | Name of the subscription plan | Bold text |
| Price | Plan price in INR | Bold, formatted |
| Duration | Monthly or Yearly | Colored pill badge |
| Max Posts | Maximum posts per month | Center-aligned |
| Max Accounts | Maximum social accounts | Center-aligned |
| AI Credits | AI credits included | Center-aligned |
| Status | Active or Inactive | Green/Red badge |
| Actions | Edit and Delete buttons | Icon buttons |

### Status Badge Colors

- **Active**: Green badge (`bg-success`)
- **Inactive**: Red badge (`bg-danger`)

### Duration Badge Colors

- **Monthly**: Green badge (`bg-success`)
- **Yearly**: Amber badge (`bg-warning`)

## TailAdmin Classes Reference

### Card Container
```html
<div class="rounded-sm border border-stroke bg-white shadow-default dark:border-strokedark dark:bg-boxdark">
```

### Table Header
```html
<th class="min-w-[150px] px-4 py-4 font-medium text-black dark:text-white">
```

### Table Cell
```html
<td class="px-4 py-5 text-black dark:text-white">
```

### Status Badge
```html
<span class="inline-flex rounded-full bg-opacity-10 px-3 py-1 text-sm font-medium bg-success text-success">
```

### Primary Button
```html
<button class="inline-flex items-center justify-center gap-2.5 rounded-lg bg-primary py-2.5 px-6 text-center font-medium text-white hover:bg-opacity-90">
```

### Form Input
```html
<input class="w-full rounded-lg border border-stroke bg-transparent py-3 px-5 outline-none focus:border-primary dark:border-form-strokedark dark:bg-form-input dark:text-white dark:focus:border-primary" />
```

## JavaScript Functions

### Modal Functions

```javascript
openPlanModal(type, planId = null)
// Opens create or edit modal
// type: 'create' or 'edit'
// planId: Required only for edit

closePlanModal()
// Closes the plan modal

openPlanDeleteModal(planId)
// Opens delete confirmation modal
// planId: ID of plan to delete

closePlanDeleteModal()
// Closes the delete confirmation modal
```

### Search Function
```javascript
// Automatically filters table rows based on search input
// Searches across all visible text in the table row
```

### Keyboard Shortcuts
- **ESC**: Close any open modal
- **Background Click**: Close modal when clicking outside

## Backend Validation

The `PlanController` validates:

```php
'name' => 'required|string|max:100',
'price' => 'required|numeric|min:0',
'duration' => 'required|in:monthly,yearly',
'max_posts_per_month' => 'required|integer|min:0',
'max_social_accounts' => 'required|integer|min:0',
'ai_credits' => 'required|integer|min:0',
'features' => 'nullable|string',
'status' => 'required|in:active,inactive',
```

## Database Schema

The `subscription_plans` table includes:

```sql
- id (primary key)
- name (string)
- price (decimal)
- duration (enum: monthly, yearly)
- max_posts_per_month (integer)
- max_social_accounts (integer)
- ai_credits (integer)
- features (text, nullable)
- status (enum: active, inactive)
- created_at (timestamp)
- updated_at (timestamp)
```

## Building Assets

### Development Mode
```bash
npm run dev
```

### Production Build
```bash
npm run build
```

Note: If you encounter PowerShell execution policy errors on Windows, run:
```powershell
Set-ExecutionPolicy -ExecutionPolicy RemoteSigned -Scope CurrentUser
```

Or use Command Prompt / Git Bash instead.

## Browser Compatibility

- ✅ Chrome 90+
- ✅ Firefox 88+
- ✅ Safari 14+
- ✅ Edge 90+
- ✅ Mobile browsers (iOS Safari, Chrome Mobile)

## Accessibility Features

- Semantic HTML structure
- ARIA labels on buttons
- Keyboard navigation support
- Focus states on interactive elements
- Screen reader friendly

## Performance

- Client-side search (no server requests)
- Lazy loading for modals
- Optimized SVG icons
- Minimal JavaScript dependencies
- CSS optimized with Tailwind purge

## Customization

### Change Primary Color
Edit `tailwind.config.js`:
```javascript
colors: {
    primary: '#YOUR_COLOR', // Change this
}
```

### Adjust Pagination
Edit `app/Http/Controllers/Admin/PlanController.php`:
```php
$plans = SubscriptionPlan::orderBy('id', 'asc')->paginate(20); // Change 20
```

### Add More Fields
1. Add migration column
2. Update form in `index.blade.php`
3. Add validation in `PlanController.php`
4. Add table column in view

## Troubleshooting

### Modal Not Opening
- Check browser console for JavaScript errors
- Ensure Alpine.js and jQuery are loaded
- Verify modal IDs match JavaScript functions

### Styles Not Applied
- Run `npm run build` to compile assets
- Clear browser cache
- Check if Vite manifest exists in `public/build/`

### Search Not Working
- Check if table has ID `plans-table`
- Verify search input has ID `search-plans`
- Check browser console for errors

### Dark Mode Issues
- Verify `dark` class is on `<html>` or `<body>` element
- Check if dark mode classes are in Tailwind config
- Rebuild assets with `npm run build`

## Credits

- **Design System**: [TailAdmin](https://tailadmin.com) - Open-source Tailwind CSS Admin Template
- **CSS Framework**: [Tailwind CSS](https://tailwindcss.com)
- **Icons**: Custom SVG icons from TailAdmin
- **Framework**: Laravel 11 with Blade templating

## Support

For TailAdmin documentation and resources:
- Website: https://tailadmin.com
- Demo: https://demo.tailadmin.com
- Documentation: https://tailadmin.com/docs
- Components: https://tailadmin.com/components

## Version

- **TailAdmin**: v2.2
- **Laravel**: 11.x
- **Tailwind CSS**: 3.4.13
- **Last Updated**: October 2025

