# Password Reset Flow - Complete Explanation

## Question 1: Why does wrong email return success with debug: null?

### Current Behavior

When you provide a **wrong/non-existent email**:

```json
{
    "success": true,
    "message": "If an account exists with that email, we have sent a password reset link.",
    "debug": null
}
```

When you provide a **correct/existing email**:

```json
{
    "success": true,
    "message": "If an account exists with that email, we have sent a password reset link.",
    "debug": {
        "token": "abc123...",
        "reset_url": "..."
    }
}
```

### Why This Design?

**This is a SECURITY FEATURE, not a bug!**

1. **Prevents Email Enumeration Attacks**: If the API told you "email doesn't exist", attackers could:

    - Try different emails to find which ones are registered
    - Build a list of valid user emails
    - Target those users with spam/phishing

2. **Privacy Protection**: Users' email addresses shouldn't be discoverable through the API

3. **Industry Standard**: This is how major platforms (Gmail, Facebook, etc.) handle password resets

### How to Know if Email Was Sent in Production?

**Short Answer: You CAN'T and SHOULDN'T know from the API response.**

**Why?** Because if you could tell, so could attackers!

**What Actually Happens:**

-   ✅ If email exists → Token is generated → Email is sent (in production)
-   ❌ If email doesn't exist → No token generated → No email sent → Same response returned

**In Production:**

-   The `debug` field is **completely removed**
-   Response is always the same regardless of email existence
-   User checks their email inbox to know if they received the reset link
-   This is the **correct and secure** way to handle it

### For Testing/Development

In development mode, you can use the `debug` field to:

-   See if email exists (`debug` is null = email doesn't exist)
-   Get the token for testing
-   Test the reset password flow

**But remember**: Remove `debug` completely in production!

---

## Question 2: Reset Password URL Flow

### Complete Flow Diagram

```
1. User requests forgot password
   ↓
2. Backend generates token and sends email
   ↓
3. Email contains link: http://frontend.com/reset-password?token=xxx&email=xxx
   ↓
4. User clicks link (GET request - opens frontend page)
   ↓
5. Frontend page displays password reset form
   ↓
6. User enters new password and submits form
   ↓
7. Frontend makes POST request to /api/reset-password
   ↓
8. Backend validates and resets password
```

### Step-by-Step Breakdown

#### Step 1-2: Forgot Password Request

**Backend**: `POST /api/forgot-password`

-   Generates token
-   Stores token in database
-   **Sends email** with reset link (in production)

#### Step 3: Email Content

The email should contain a link like:

```
http://your-frontend-domain.com/reset-password?token=abc123def456&email=user@example.com
```

**Important**: This is a **GET request** that opens your **Next.js frontend page**, NOT the API endpoint!

#### Step 4: User Clicks Link

-   Browser makes **GET request** to frontend URL
-   Opens Next.js page: `/reset-password`
-   URL parameters (`token` and `email`) are available in the page

#### Step 5: Frontend Page Displays Form

The Next.js page should:

-   Extract `token` and `email` from URL query parameters
-   Display a form with:
    -   Password field
    -   Password confirmation field
    -   Submit button

#### Step 6-7: User Submits Form

-   Frontend makes **POST request** to: `POST /api/reset-password`
-   Sends JSON body:

```json
{
    "email": "user@example.com",
    "token": "abc123def456",
    "password": "newpassword123",
    "password_confirmation": "newpassword123"
}
```

#### Step 8: Backend Processes

-   Validates token
-   Checks expiration (60 minutes)
-   Updates password
-   Returns success/error response

---

## Email Link Structure

### What the Email Should Contain

**Subject**: "Reset Your Password"

**Body**:

```
Hello,

You requested to reset your password. Click the link below to reset it:

[Reset Password](http://your-frontend.com/reset-password?token=TOKEN_HERE&email=EMAIL_HERE)

This link will expire in 60 minutes.

If you didn't request this, please ignore this email.
```

### URL Format

```
http://FRONTEND_DOMAIN/reset-password?token=TOKEN&email=EMAIL
```

**Example**:

```
http://localhost:3000/reset-password?token=abc123def456ghi789&email=user%40example.com
```

**Note**:

-   `FRONTEND_DOMAIN` = Your Next.js app URL (NOT the Laravel API URL)
-   `token` = The plain token (not hashed)
-   `email` = URL encoded email address

---

## Next.js Frontend Implementation

### Reset Password Page (`/pages/reset-password.js` or `/app/reset-password/page.js`)

```javascript
import { useRouter } from "next/router";
import { useState, useEffect } from "react";

export default function ResetPassword() {
    const router = useRouter();
    const { token, email } = router.query; // GET parameters from URL

    const [password, setPassword] = useState("");
    const [passwordConfirmation, setPasswordConfirmation] = useState("");
    const [loading, setLoading] = useState(false);
    const [message, setMessage] = useState("");
    const [errors, setErrors] = useState({});
    const [tokenValid, setTokenValid] = useState(null);

    // Check if token and email are present
    useEffect(() => {
        if (!token || !email) {
            setMessage(
                "Invalid reset link. Please request a new password reset."
            );
            setTokenValid(false);
        } else {
            setTokenValid(true);
        }
    }, [token, email]);

    const handleSubmit = async (e) => {
        e.preventDefault();
        setLoading(true);
        setMessage("");
        setErrors({});

        // Frontend validation
        if (password !== passwordConfirmation) {
            setErrors({ password: ["Passwords do not match"] });
            setLoading(false);
            return;
        }

        if (password.length < 6) {
            setErrors({ password: ["Password must be at least 6 characters"] });
            setLoading(false);
            return;
        }

        try {
            // POST request to API
            const response = await fetch(
                "http://localhost:8000/api/reset-password",
                {
                    method: "POST",
                    headers: {
                        "Content-Type": "application/json",
                        Accept: "application/json",
                    },
                    body: JSON.stringify({
                        email: email,
                        token: token,
                        password: password,
                        password_confirmation: passwordConfirmation,
                    }),
                }
            );

            const data = await response.json();

            if (data.success) {
                setMessage(
                    "Password reset successfully! Redirecting to login..."
                );
                // Redirect to login after 2 seconds
                setTimeout(() => {
                    router.push("/login");
                }, 2000);
            } else {
                setErrors(data.errors || {});
                setMessage(data.message || "An error occurred");
            }
        } catch (error) {
            setMessage("An error occurred. Please try again.");
            console.error("Reset password error:", error);
        } finally {
            setLoading(false);
        }
    };

    if (tokenValid === false) {
        return (
            <div>
                <h1>Invalid Reset Link</h1>
                <p>{message}</p>
                <a href="/forgot-password">Request New Reset Link</a>
            </div>
        );
    }

    return (
        <div>
            <h1>Reset Your Password</h1>
            <form onSubmit={handleSubmit}>
                <div>
                    <label>New Password</label>
                    <input
                        type="password"
                        value={password}
                        onChange={(e) => setPassword(e.target.value)}
                        required
                        minLength={6}
                    />
                    {errors.password && (
                        <div style={{ color: "red" }}>
                            {errors.password.map((err, i) => (
                                <p key={i}>{err}</p>
                            ))}
                        </div>
                    )}
                </div>

                <div>
                    <label>Confirm Password</label>
                    <input
                        type="password"
                        value={passwordConfirmation}
                        onChange={(e) =>
                            setPasswordConfirmation(e.target.value)
                        }
                        required
                        minLength={6}
                    />
                </div>

                <button type="submit" disabled={loading}>
                    {loading ? "Resetting..." : "Reset Password"}
                </button>

                {message && (
                    <div
                        style={{
                            color: message.includes("success")
                                ? "green"
                                : "red",
                        }}
                    >
                        {message}
                    </div>
                )}
            </form>
        </div>
    );
}
```

---

## Summary

### Forgot Password Endpoint

-   **Method**: POST
-   **URL**: `/api/forgot-password`
-   **Response**: Always success (security feature)
-   **Debug**: Only in development (shows if email exists)

### Email Link

-   **Type**: GET request
-   **URL**: `http://FRONTEND/reset-password?token=xxx&email=xxx`
-   **Purpose**: Opens Next.js frontend page
-   **NOT**: An API endpoint

### Reset Password Endpoint

-   **Method**: POST
-   **URL**: `/api/reset-password`
-   **Called by**: Frontend form submission
-   **Body**: `{ email, token, password, password_confirmation }`

### Key Points

1. ✅ Email link opens **frontend page** (GET request)
2. ✅ Frontend page makes **POST request** to API
3. ✅ Can't tell if email exists from API response (security feature)
4. ✅ In production, remove `debug` field completely
5. ✅ User checks email inbox to know if reset link was sent
