# Next.js Frontend API Integration Guide

## Overview
This guide explains how to integrate your Next.js frontend with the Laravel API backend, including authentication, token management, and API route configuration.

## Table of Contents
1. [API Route Configuration](#api-route-configuration)
2. [How Sanctum Tokens Work](#how-sanctum-tokens-work)
3. [Token Storage Options](#token-storage-options)
4. [Next.js Setup](#nextjs-setup)
5. [API Client Configuration](#api-client-configuration)
6. [Authentication Flow](#authentication-flow)
7. [Complete Code Examples](#complete-code-examples)

---

## API Route Configuration

### Understanding Laravel API Routes

#### 1. Route File Structure
All API routes are defined in `routes/api.php`:

```php
<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;

// Public routes (no authentication)
Route::post('/register', [AuthController::class, 'register']);
Route::post('/login', [AuthController::class, 'login']);

// Protected routes (require authentication token)
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/user', [AuthController::class, 'profile']);
    Route::post('/logout', [AuthController::class, 'logout']);
    // ... other protected routes
});
```

#### 2. How Routes Work

**Route Definition:**
```php
Route::post('/login', [AuthController::class, 'login']);
```

**What This Means:**
- **HTTP Method**: `POST`
- **URL Path**: `/api/login` (Laravel automatically prefixes `/api` to all routes in `api.php`)
- **Controller**: `App\Http\Controllers\Api\AuthController`
- **Method**: `login()`

**Full URL Examples:**
- Local: `http://localhost:8000/api/login`
- Production: `https://yourdomain.com/api/login`

#### 3. Route Middleware

**Public Routes** (No authentication):
```php
Route::post('/login', [AuthController::class, 'login']);
```
- Anyone can access these routes
- No token required

**Protected Routes** (Authentication required):
```php
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/user', [AuthController::class, 'profile']);
});
```
- Requires valid Sanctum token in `Authorization` header
- Returns 401 Unauthorized if token is missing or invalid

---

## How Sanctum Tokens Work

### What is Laravel Sanctum?

Laravel Sanctum provides a simple authentication system for SPAs (Single Page Applications) and mobile applications using **token-based authentication**.

### Token Lifecycle

#### 1. Token Creation (Login)
```php
// In AuthController@login
$token = $user->createToken('web_app_token')->plainTextToken;
```

**What Happens:**
- A new token is created in the `personal_access_tokens` table
- Token is associated with the user
- Token name is stored (e.g., 'web_app_token')
- Returns a plain text token: `1|abcdef1234567890...`

**Token Format:**
```
{token_id}|{hashed_token}
```

#### 2. Token Storage in Database

**Table**: `personal_access_tokens`

**Structure:**
- `id`: Token ID
- `tokenable_type`: Model class (e.g., `App\Models\User`)
- `tokenable_id`: User ID
- `name`: Token name (e.g., 'web_app_token')
- `token`: Hashed token (not the plain text)
- `abilities`: Permissions (usually `["*"]` for all)
- `last_used_at`: Last usage timestamp
- `expires_at`: Expiration date (optional)
- `created_at`: Creation timestamp

**Important:** The token stored in the database is **hashed**. The plain text token is only returned once during creation.

#### 3. Token Validation (Protected Routes)

When you make a request with a token:
```javascript
Authorization: Bearer 1|abcdef1234567890...
```

**What Happens:**
1. Laravel extracts the token from the header
2. Parses the token ID (before `|`)
3. Looks up the token in `personal_access_tokens` table
4. Hashes the provided token and compares with stored hash
5. If valid, loads the associated user
6. Makes user available via `$request->user()`

#### 4. Token Revocation (Logout)
```php
// In AuthController@logout
$request->user()->currentAccessToken()->delete();
```

**What Happens:**
- Deletes the token from `personal_access_tokens` table
- Token becomes invalid immediately
- User must login again to get a new token

### Token Security Features

1. **Hashing**: Tokens are hashed before storage (one-way encryption)
2. **Unique**: Each token is unique and cannot be reverse-engineered
3. **Revocable**: Tokens can be deleted to invalidate them
4. **Expirable**: Optional expiration dates can be set
5. **Scoped**: Tokens can have specific abilities/permissions

---

## Token Storage Options

### Option 1: localStorage (Recommended for Next.js)

**Pros:**
- ✅ Persists across browser sessions
- ✅ Survives page refreshes
- ✅ Easy to access in client-side code
- ✅ Works with Next.js client components

**Cons:**
- ❌ Vulnerable to XSS attacks (if your site has XSS vulnerabilities)
- ❌ Accessible via JavaScript (can be stolen if malicious script runs)

**Best For:**
- Next.js applications
- SPAs (Single Page Applications)
- When you have good XSS protection

**Implementation:**
```typescript
// Store token
localStorage.setItem('auth_token', token);

// Retrieve token
const token = localStorage.getItem('auth_token');

// Remove token
localStorage.removeItem('auth_token');
```

### Option 2: sessionStorage

**Pros:**
- ✅ Automatically cleared when tab closes
- ✅ More secure than localStorage (shorter lifetime)
- ✅ Easy to access in client-side code

**Cons:**
- ❌ Lost when tab closes
- ❌ User must login again if they close the tab
- ❌ Still vulnerable to XSS

**Best For:**
- Temporary sessions
- When you want automatic logout on tab close

**Implementation:**
```typescript
// Store token
sessionStorage.setItem('auth_token', token);

// Retrieve token
const token = sessionStorage.getItem('auth_token');

// Remove token
sessionStorage.removeItem('auth_token');
```

### Option 3: httpOnly Cookies (Most Secure)

**Pros:**
- ✅ Most secure (not accessible via JavaScript)
- ✅ Protected from XSS attacks
- ✅ Automatically sent with requests
- ✅ Can be httpOnly and secure flags

**Cons:**
- ❌ Requires backend configuration
- ❌ More complex setup
- ❌ CORS considerations
- ❌ Not accessible in client-side JavaScript

**Best For:**
- High-security applications
- When you need maximum security
- Server-side rendering scenarios

**Note:** This requires modifying the Laravel backend to use cookie-based authentication instead of token-based.

### Recommendation for Next.js

**Use localStorage** because:
1. Next.js is a client-side framework
2. You need easy access to tokens in client components
3. Modern browsers have good XSS protection
4. You can implement additional security measures (token refresh, expiration checks)

**Security Best Practices:**
1. Always use HTTPS in production
2. Implement token expiration
3. Add token refresh mechanism
4. Validate tokens on the backend
5. Use Content Security Policy (CSP) headers
6. Sanitize user inputs to prevent XSS

---

## Next.js Setup

### 1. Install Dependencies

```bash
npm install axios
# or
yarn add axios
```

### 2. Create API Configuration

Create `lib/api.ts`:

```typescript
import axios from 'axios';

// Base URL for your Laravel API
const API_BASE_URL = process.env.NEXT_PUBLIC_API_URL || 'http://localhost:8000';

// Create axios instance
const apiClient = axios.create({
  baseURL: `${API_BASE_URL}/api`,
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
});

// Request interceptor - Add token to requests
apiClient.interceptors.request.use(
  (config) => {
    // Get token from localStorage
    if (typeof window !== 'undefined') {
      const token = localStorage.getItem('auth_token');
      if (token) {
        config.headers.Authorization = `Bearer ${token}`;
      }
    }
    return config;
  },
  (error) => {
    return Promise.reject(error);
  }
);

// Response interceptor - Handle errors
apiClient.interceptors.response.use(
  (response) => response,
  (error) => {
    // Handle 401 Unauthorized (token expired/invalid)
    if (error.response?.status === 401) {
      // Clear token and redirect to login
      if (typeof window !== 'undefined') {
        localStorage.removeItem('auth_token');
        localStorage.removeItem('user');
        window.location.href = '/login';
      }
    }
    return Promise.reject(error);
  }
);

export default apiClient;
```

### 3. Create Auth Service

Create `lib/auth.ts`:

```typescript
import apiClient from './api';

export interface LoginCredentials {
  email: string;
  password: string;
}

export interface User {
  id: number;
  name: string;
  email: string;
  role: string;
  timezone: string;
  status: string;
}

export interface LoginResponse {
  success: boolean;
  message: string;
  data: {
    token: string;
    user: User;
  };
}

export const authService = {
  /**
   * Login user
   */
  async login(credentials: LoginCredentials): Promise<LoginResponse> {
    const response = await apiClient.post<LoginResponse>('/login', credentials);
    
    if (response.data.success && response.data.data.token) {
      // Store token and user data
      if (typeof window !== 'undefined') {
        localStorage.setItem('auth_token', response.data.data.token);
        localStorage.setItem('user', JSON.stringify(response.data.data.user));
      }
    }
    
    return response.data;
  },

  /**
   * Logout user
   */
  async logout(): Promise<void> {
    try {
      await apiClient.post('/logout');
    } catch (error) {
      console.error('Logout error:', error);
    } finally {
      // Clear local storage regardless of API call success
      if (typeof window !== 'undefined') {
        localStorage.removeItem('auth_token');
        localStorage.removeItem('user');
      }
    }
  },

  /**
   * Get current user
   */
  async getCurrentUser(): Promise<User | null> {
    try {
      const response = await apiClient.get<{ success: boolean; data: User }>('/user');
      return response.data.data;
    } catch (error) {
      return null;
    }
  },

  /**
   * Get stored user from localStorage
   */
  getStoredUser(): User | null {
    if (typeof window === 'undefined') return null;
    
    const userStr = localStorage.getItem('user');
    if (!userStr) return null;
    
    try {
      return JSON.parse(userStr);
    } catch {
      return null;
    }
  },

  /**
   * Check if user is authenticated
   */
  isAuthenticated(): boolean {
    if (typeof window === 'undefined') return false;
    return !!localStorage.getItem('auth_token');
  },

  /**
   * Get stored token
   */
  getToken(): string | null {
    if (typeof window === 'undefined') return null;
    return localStorage.getItem('auth_token');
  },
};
```

---

## Authentication Flow

### Complete Login Flow

```
1. User enters email/password
   ↓
2. Frontend calls POST /api/login
   ↓
3. Laravel validates credentials
   ↓
4. Laravel creates token and returns it
   ↓
5. Frontend stores token in localStorage
   ↓
6. Frontend stores user data in localStorage
   ↓
7. Frontend redirects to dashboard
```

### Protected Route Access Flow

```
1. User navigates to protected page
   ↓
2. Frontend checks localStorage for token
   ↓
3. If token exists, add to Authorization header
   ↓
4. Make API request with token
   ↓
5. Laravel validates token
   ↓
6. If valid: Return data
   If invalid: Return 401
   ↓
7. Frontend handles response
   If 401: Clear token and redirect to login
```

---

## Complete Code Examples

### 1. Login Page Component

Create `app/login/page.tsx` (App Router) or `pages/login.tsx` (Pages Router):

```typescript
'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';
import { authService } from '@/lib/auth';

export default function LoginPage() {
  const router = useRouter();
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setError('');
    setLoading(true);

    try {
      const response = await authService.login({ email, password });
      
      if (response.success) {
        router.push('/dashboard');
      } else {
        setError(response.message || 'Login failed');
      }
    } catch (err: any) {
      const errorMessage = err.response?.data?.message || 'An error occurred';
      setError(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gray-50">
      <div className="max-w-md w-full space-y-8 p-8 bg-white rounded-lg shadow">
        <h2 className="text-2xl font-bold text-center">Login</h2>
        
        <form onSubmit={handleSubmit} className="space-y-4">
          {error && (
            <div className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
              {error}
            </div>
          )}
          
          <div>
            <label htmlFor="email" className="block text-sm font-medium text-gray-700">
              Email
            </label>
            <input
              id="email"
              type="email"
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              required
              className="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500"
            />
          </div>
          
          <div>
            <label htmlFor="password" className="block text-sm font-medium text-gray-700">
              Password
            </label>
            <input
              id="password"
              type="password"
              value={password}
              onChange={(e) => setPassword(e.target.value)}
              required
              className="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500"
            />
          </div>
          
          <button
            type="submit"
            disabled={loading}
            className="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:opacity-50"
          >
            {loading ? 'Logging in...' : 'Login'}
          </button>
        </form>
      </div>
    </div>
  );
}
```

### 2. Protected Route Middleware

Create `middleware.ts` (App Router):

```typescript
import { NextResponse } from 'next/server';
import type { NextRequest } from 'next/server';

export function middleware(request: NextRequest) {
  // Check if route requires authentication
  const protectedPaths = ['/dashboard', '/profile', '/posts'];
  const isProtectedPath = protectedPaths.some(path => 
    request.nextUrl.pathname.startsWith(path)
  );

  if (isProtectedPath) {
    // Check for token in cookie or header
    const token = request.cookies.get('auth_token')?.value || 
                  request.headers.get('authorization')?.replace('Bearer ', '');

    if (!token) {
      // Redirect to login if no token
      return NextResponse.redirect(new URL('/login', request.url));
    }
  }

  return NextResponse.next();
}

export const config = {
  matcher: ['/dashboard/:path*', '/profile/:path*', '/posts/:path*'],
};
```

### 3. Protected Page Component

Create `app/dashboard/page.tsx`:

```typescript
'use client';

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import { authService, User } from '@/lib/auth';

export default function DashboardPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    // Check authentication
    if (!authService.isAuthenticated()) {
      router.push('/login');
      return;
    }

    // Load user data
    const loadUser = async () => {
      const userData = await authService.getCurrentUser();
      if (userData) {
        setUser(userData);
      } else {
        router.push('/login');
      }
      setLoading(false);
    };

    loadUser();
  }, [router]);

  const handleLogout = async () => {
    await authService.logout();
    router.push('/login');
  };

  if (loading) {
    return <div>Loading...</div>;
  }

  if (!user) {
    return null;
  }

  return (
    <div className="min-h-screen bg-gray-50 p-8">
      <div className="max-w-7xl mx-auto">
        <div className="flex justify-between items-center mb-8">
          <h1 className="text-3xl font-bold">Dashboard</h1>
          <button
            onClick={handleLogout}
            className="px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
          >
            Logout
          </button>
        </div>
        
        <div className="bg-white rounded-lg shadow p-6">
          <h2 className="text-xl font-semibold mb-4">User Information</h2>
          <div className="space-y-2">
            <p><strong>Name:</strong> {user.name}</p>
            <p><strong>Email:</strong> {user.email}</p>
            <p><strong>Role:</strong> {user.role}</p>
            <p><strong>Status:</strong> {user.status}</p>
          </div>
        </div>
      </div>
    </div>
  );
}
```

### 4. Environment Variables

Create `.env.local`:

```env
NEXT_PUBLIC_API_URL=http://localhost:8000
```

For production:
```env
NEXT_PUBLIC_API_URL=https://your-api-domain.com
```

---

## CORS Configuration

### Laravel CORS Setup

If your Next.js app runs on a different domain/port, configure CORS in Laravel:

1. Install CORS package (usually included):
```bash
composer require fruitcake/laravel-cors
```

2. Configure `config/cors.php`:

```php
<?php

return [
    'paths' => ['api/*', 'sanctum/csrf-cookie'],
    'allowed_methods' => ['*'],
    'allowed_origins' => [
        'http://localhost:3000', // Next.js dev server
        'https://your-nextjs-domain.com', // Production
    ],
    'allowed_origins_patterns' => [],
    'allowed_headers' => ['*'],
    'exposed_headers' => [],
    'max_age' => 0,
    'supports_credentials' => true,
];
```

---

## Testing the Integration

### 1. Test Login

```bash
curl -X POST http://localhost:8000/api/login \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "email": "user@example.com",
    "password": "password123"
  }'
```

### 2. Test Protected Route

```bash
# First, get token from login response
TOKEN="1|abcdef1234567890..."

curl -X GET http://localhost:8000/api/user \
  -H "Authorization: Bearer $TOKEN" \
  -H "Accept: application/json"
```

---

## Summary

### Key Points:

1. **API Routes**: Defined in `routes/api.php`, automatically prefixed with `/api`
2. **Tokens**: Created on login, stored in database (hashed), returned as plain text once
3. **Storage**: Use `localStorage` for Next.js (recommended)
4. **Security**: Tokens are hashed in DB, validate on each request
5. **Flow**: Login → Get Token → Store Token → Use Token in Headers → Logout → Delete Token

### Token Storage Decision:

**✅ Use localStorage** for Next.js because:
- Easy access in client components
- Persists across sessions
- Standard practice for SPAs
- Good security with HTTPS + proper XSS protection

**❌ Don't store in database** - The token is already stored in Laravel's `personal_access_tokens` table. You only need to store it in localStorage to send it with requests.

---

## Troubleshooting

### Common Issues:

1. **CORS Errors**: Configure `config/cors.php` in Laravel
2. **401 Unauthorized**: Check token is in `Authorization: Bearer {token}` header
3. **Token Not Found**: Verify token is stored in localStorage
4. **Network Errors**: Check API URL in `.env.local`

---

## Next Steps

1. Set up your Next.js project
2. Install axios: `npm install axios`
3. Copy the code examples above
4. Configure environment variables
5. Test the login flow
6. Implement protected routes
7. Add error handling
8. Set up token refresh mechanism (optional)

