# Mobile API Login Guide

## Overview
This guide explains how to use the login API endpoint for mobile applications.

## API Endpoint

### Login URL
```
POST /api/login
```

**Full URL Examples:**
- Local Development: `http://localhost:8000/api/login`
- Production: `https://yourdomain.com/api/login`

## Request Format

### Headers
```
Content-Type: application/json
Accept: application/json
```

### Request Body
```json
{
    "email": "user@example.com",
    "password": "password123"
}
```

### Required Fields
- `email` (string, required): Valid email address
- `password` (string, required): Minimum 6 characters

## Response Format

### Success Response (200 OK)
```json
{
    "success": true,
    "message": "Login successful.",
    "data": {
        "token": "1|abcdef1234567890...",
        "user": {
            "id": 1,
            "name": "John Doe",
            "email": "user@example.com",
            "role": "user",
            "timezone": "UTC",
            "status": "active"
        }
    }
}
```

### Validation Error Response (422 Unprocessable Entity)
```json
{
    "message": "The email field is required. (and 1 more error)",
    "errors": {
        "email": [
            "Email address is required."
        ],
        "password": [
            "Password is required."
        ]
    }
}
```

### Authentication Error Response (401 Unauthorized)
```json
{
    "success": false,
    "message": "Invalid email or password.",
    "errors": {
        "email": [
            "The provided credentials do not match our records."
        ]
    }
}
```

### Account Inactive Response (403 Forbidden)
```json
{
    "success": false,
    "message": "Your account is not active. Please contact support."
}
```

## Using the Token

After successful login, you'll receive a token. Use this token to authenticate subsequent API requests.

### Adding Token to Requests
Include the token in the `Authorization` header:

```
Authorization: Bearer {your_token_here}
```

### Example: Get User Profile
```
GET /api/user
Headers:
  Authorization: Bearer 1|abcdef1234567890...
  Accept: application/json
```

## Code Examples

### JavaScript/React Native (Fetch API)
```javascript
const login = async (email, password) => {
  try {
    const response = await fetch('http://localhost:8000/api/login', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
      body: JSON.stringify({
        email: email,
        password: password,
      }),
    });

    const data = await response.json();

    if (data.success) {
      // Store token securely
      await AsyncStorage.setItem('auth_token', data.data.token);
      // Store user data
      await AsyncStorage.setItem('user', JSON.stringify(data.data.user));
      return data;
    } else {
      throw new Error(data.message);
    }
  } catch (error) {
    console.error('Login error:', error);
    throw error;
  }
};
```

### Android (Kotlin)
```kotlin
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.RequestBody.Companion.toRequestBody
import org.json.JSONObject

fun login(email: String, password: String, callback: (Result<JSONObject>) -> Unit) {
    val client = OkHttpClient()
    
    val json = JSONObject()
    json.put("email", email)
    json.put("password", password)
    
    val requestBody = json.toString()
        .toRequestBody("application/json".toMediaType())
    
    val request = Request.Builder()
        .url("http://localhost:8000/api/login")
        .post(requestBody)
        .addHeader("Content-Type", "application/json")
        .addHeader("Accept", "application/json")
        .build()
    
    client.newCall(request).enqueue(object : Callback {
        override fun onFailure(call: Call, e: IOException) {
            callback(Result.failure(e))
        }
        
        override fun onResponse(call: Call, response: Response) {
            val responseBody = response.body?.string()
            if (response.isSuccessful && responseBody != null) {
                val jsonResponse = JSONObject(responseBody)
                callback(Result.success(jsonResponse))
            } else {
                callback(Result.failure(Exception("Login failed")))
            }
        }
    })
}
```

### iOS (Swift)
```swift
import Foundation

func login(email: String, password: String, completion: @escaping (Result<[String: Any], Error>) -> Void) {
    let url = URL(string: "http://localhost:8000/api/login")!
    var request = URLRequest(url: url)
    request.httpMethod = "POST"
    request.setValue("application/json", forHTTPHeaderField: "Content-Type")
    request.setValue("application/json", forHTTPHeaderField: "Accept")
    
    let body: [String: Any] = [
        "email": email,
        "password": password
    ]
    
    request.httpBody = try? JSONSerialization.data(withJSONObject: body)
    
    URLSession.shared.dataTask(with: request) { data, response, error in
        if let error = error {
            completion(.failure(error))
            return
        }
        
        guard let data = data,
              let json = try? JSONSerialization.jsonObject(with: data) as? [String: Any] else {
            completion(.failure(NSError(domain: "Invalid response", code: 0)))
            return
        }
        
        completion(.success(json))
    }.resume()
}
```

## API Route Structure

### How Laravel API Routes Work

1. **Route File**: `routes/api.php`
   - All API routes are defined here
   - Routes are automatically prefixed with `/api`

2. **Route Definition**:
```php
Route::post('/login', [AuthController::class, 'login']);
```
   - This creates: `POST /api/login`
   - Maps to: `AuthController@login` method

3. **Controller**: `app/Http/Controllers/Api/AuthController.php`
   - Handles the login logic
   - Uses `LoginRequest` for validation

4. **Validation**: `app/Http/Requests/Api/LoginRequest.php`
   - Validates email and password
   - Returns proper error messages

## Testing the API

### Using cURL
```bash
curl -X POST http://localhost:8000/api/login \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "email": "user@example.com",
    "password": "password123"
  }'
```

### Using Postman
1. Create a new POST request
2. URL: `http://localhost:8000/api/login`
3. Headers:
   - `Content-Type: application/json`
   - `Accept: application/json`
4. Body (raw JSON):
```json
{
    "email": "user@example.com",
    "password": "password123"
}
```

## Security Notes

1. **Always use HTTPS** in production
2. **Store tokens securely** (use secure storage like Keychain/Keystore)
3. **Never log or expose tokens** in client-side code
4. **Implement token refresh** for long-lived sessions
5. **Handle token expiration** gracefully

## Troubleshooting

### Common Issues

1. **401 Unauthorized**: Check email/password are correct
2. **422 Validation Error**: Ensure all required fields are provided
3. **403 Forbidden**: User account may be inactive
4. **CORS Issues**: Configure CORS in `config/cors.php` for mobile apps
5. **Token Not Working**: Ensure token is included in `Authorization: Bearer {token}` header

## Related Endpoints

- `POST /api/register` - Register a new user
- `GET /api/user` - Get authenticated user profile (requires token)
- `POST /api/logout` - Logout (if implemented)

