# Logout API Guide

## Overview
The logout endpoint revokes the current authentication token, making it invalid for future requests.

## Endpoint Details

**URL:** `POST /api/logout`  
**Authentication:** Required (Bearer Token)  
**Route:** Protected (requires `auth:sanctum` middleware)

---

## How to Call Logout

### Step 1: Get Token (Login First)

Before logging out, you need to be logged in:

```http
POST http://localhost:8000/api/login
Content-Type: application/json

{
    "email": "ganesh@yopmail.com",
    "password": "123456"
}
```

**Response:**
```json
{
    "success": true,
    "data": {
        "token": "1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17"
    }
}
```

**Save the token** for logout request.

---

## Method 1: Using Postman

### Request Configuration

1. **Method**: `POST`
2. **URL**: `http://localhost:8000/api/logout`
3. **Headers Tab**:
   - `Authorization`: `Bearer {your_token_here}`
   - `Accept`: `application/json`

### Postman Setup

```
┌─────────────────────────────────────────┐
│ POST  http://localhost:8000/api/logout  │
├─────────────────────────────────────────┤
│ Headers                                 │
├─────────────────────────────────────────┤
│ Authorization │ Bearer 1|7eda8f53...   │
│ Accept        │ application/json        │
└─────────────────────────────────────────┘
```

### Expected Response (200 OK)

```json
{
    "success": true,
    "message": "Logged out successfully."
}
```

### What Happens

1. Token is validated
2. Token is deleted from `personal_access_tokens` table
3. Token becomes invalid immediately
4. User must login again to get a new token

---

## Method 2: Using cURL

```bash
curl -X POST http://localhost:8000/api/logout \
  -H "Authorization: Bearer 1|7eda8f53e2f0f3dcb2b52b0f196c4ed92ed21d815517ef0a7d707396cb7d0b17" \
  -H "Accept: application/json"
```

---

## Method 3: Using Next.js / JavaScript

### Setup API Client

```typescript
// lib/api.ts
import axios from 'axios';

const apiClient = axios.create({
  baseURL: process.env.NEXT_PUBLIC_API_URL + '/api',
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
});

// Automatically add token to requests
apiClient.interceptors.request.use((config) => {
  const token = localStorage.getItem('auth_token');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

export default apiClient;
```

### Logout Function

```typescript
// lib/auth.ts or your auth service
import apiClient from './api';

export const logout = async () => {
  try {
    // Call logout API (token automatically added by interceptor)
    await apiClient.post('/logout');
  } catch (error) {
    console.error('Logout error:', error);
    // Continue with cleanup even if API call fails
  } finally {
    // Always clear local storage
    localStorage.removeItem('auth_token');
    localStorage.removeItem('user');
  }
};
```

### React Component Example

```typescript
'use client';

import { useRouter } from 'next/navigation';
import { logout } from '@/lib/auth';

export default function LogoutButton() {
  const router = useRouter();

  const handleLogout = async () => {
    try {
      await logout();
      // Redirect to login page
      router.push('/login');
    } catch (error) {
      console.error('Logout failed:', error);
      // Still redirect even if logout fails
      router.push('/login');
    }
  };

  return (
    <button
      onClick={handleLogout}
      className="px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700"
    >
      Logout
    </button>
  );
}
```

### Complete Logout Hook

```typescript
// hooks/useAuth.ts
import { useState } from 'react';
import { useRouter } from 'next/navigation';
import apiClient from '@/lib/api';

export const useAuth = () => {
  const router = useRouter();
  const [loading, setLoading] = useState(false);

  const logout = async () => {
    setLoading(true);
    try {
      // Call logout API
      await apiClient.post('/logout');
    } catch (error) {
      console.error('Logout error:', error);
    } finally {
      // Clear local storage
      localStorage.removeItem('auth_token');
      localStorage.removeItem('user');
      setLoading(false);
      // Redirect to login
      router.push('/login');
    }
  };

  return { logout, loading };
};

// Usage in component
const { logout, loading } = useAuth();

<button onClick={logout} disabled={loading}>
  {loading ? 'Logging out...' : 'Logout'}
</button>
```

---

## Error Responses

### 401 Unauthorized (Missing/Invalid Token)

```json
{
    "success": false,
    "message": "Unauthenticated."
}
```

**Causes:**
- Token missing from header
- Token is invalid
- Token already revoked
- Token expired

### 500 Internal Server Error

**Possible causes:**
- Database connection issue
- Token record not found (already deleted)

---

## Important Notes

### 1. Token is Revoked Immediately
- Once logout is called, the token is deleted from database
- Token cannot be used for any future requests
- User must login again to get a new token

### 2. Multiple Tokens
- Users can have multiple tokens (different devices)
- Logout only revokes the **current token** used in the request
- Other tokens remain valid

### 3. Revoke All Tokens (Optional)

If you want to logout from all devices, you can modify the logout method:

```php
// In AuthController@logout
public function logout(Request $request)
{
    // Revoke ALL tokens for this user
    $request->user()->tokens()->delete();

    return response()->json([
        'success' => true,
        'message' => 'Logged out successfully from all devices.'
    ]);
}
```

### 4. Frontend Cleanup

Always clear local storage after logout:

```typescript
// Clear token and user data
localStorage.removeItem('auth_token');
localStorage.removeItem('user');
```

---

## Complete Logout Flow

```
1. User clicks logout button
   ↓
2. Frontend calls POST /api/logout
   ↓
3. Token sent in Authorization header
   ↓
4. Laravel validates token
   ↓
5. Deletes token from database
   ↓
6. Returns success response
   ↓
7. Frontend clears localStorage
   ↓
8. Frontend redirects to login page
```

---

## Testing Checklist

- [ ] Login successfully and get token
- [ ] Call logout with valid token
- [ ] Verify 200 OK response
- [ ] Try to use the same token again (should fail with 401)
- [ ] Verify token is deleted from database
- [ ] Frontend clears localStorage
- [ ] User redirected to login page

---

## Quick Reference

### Postman
```
POST http://localhost:8000/api/logout
Headers:
  Authorization: Bearer {token}
  Accept: application/json
```

### cURL
```bash
curl -X POST http://localhost:8000/api/logout \
  -H "Authorization: Bearer {token}" \
  -H "Accept: application/json"
```

### Next.js
```typescript
await apiClient.post('/logout');
localStorage.removeItem('auth_token');
localStorage.removeItem('user');
router.push('/login');
```

---

## Summary

✅ **Endpoint**: `POST /api/logout`  
✅ **Auth Required**: Yes (Bearer Token)  
✅ **Response**: `{"success": true, "message": "Logged out successfully."}`  
✅ **Action**: Deletes token from database  
✅ **Next Step**: Clear localStorage and redirect to login

The logout function is ready to use! Just send a POST request with your token in the Authorization header.

