# Laravel API Implementation Guide - Posts & Drafts System

## 📋 Overview

This document provides complete API specifications for implementing the Posts and Drafts system in Laravel. It includes all endpoints, request/response formats, validation rules, and implementation examples.

**Base URL:** `http://your-domain.com/api`

**Authentication:** All endpoints require Bearer token authentication:

```
Authorization: Bearer {token}
```

---

## 🔑 Key Concept: Identifying Draft vs Publish

### Problem Statement

The frontend sends the same payload structure for both "Save Draft" and "Publish" actions. The backend needs to distinguish between them.

### Solution

1. **Save Draft**: When user clicks "Save Draft", the frontend sends `status: "draft"` in the payload
2. **Publish**: When user clicks "Publish", the frontend:
    - First saves with `status: "draft"` (if not already saved)
    - Then calls the `/publish` endpoint to change status to `"posted"`

### How to Identify in Backend

**For Create/Update Endpoints (`POST /api/posts` or `PUT /api/posts/{id}`):**

-   Check the `status` field in the request payload
-   If `status === "draft"` → User clicked "Save Draft"
-   If `status === "posted"` → User wants to publish directly (optional, not currently used)

**For Publish Endpoint (`PUT /api/posts/{id}/publish`):**

-   This endpoint is ONLY called when user clicks "Publish"
-   It changes the status from `"draft"` to `"posted"`

---

## 📝 API Endpoints

### 1. Create Post (Draft)

**Endpoint:** `POST /api/posts`

**Description:** Creates a new post. The `status` field determines if it's a draft or published post.

**Request Headers:**

```
Authorization: Bearer {token}
Content-Type: application/json (or multipart/form-data if files included)
```

**Request Body (JSON - No Files):**

```json
{
    "user_id": 1,
    "content": "This is my post content",
    "content_html": "<p>This is my <strong>post</strong> content</p>",
    "status": "draft",
    "category": "General",
    "platforms": ["facebook", "twitter"]
}
```

**Request Body (Form Data - With Files):**

```
user_id: 1
content: "This is my post content"
content_html: "<p>This is my post content</p>"
status: draft
category: General
platforms: ["facebook", "twitter"]
files[]: [File 1]
files[]: [File 2]
```

**Field Descriptions:**

-   `user_id` (required): ID of the user creating the post
-   `content` (required): Plain text content of the post
-   `content_html` (optional): HTML formatted content
-   `status` (required): Either `"draft"` or `"posted"` - **This is how you identify the action!**
    -   `"draft"` = User clicked "Save Draft"
    -   `"posted"` = User wants to publish directly (optional feature)
-   `category` (optional): Post category (default: "General")
-   `platforms` (optional): JSON array of platform names
-   `files[]` (optional): Array of media files (images/videos)

**Validation Rules:**

```php
[
    'user_id' => 'required|exists:users,id',
    'content' => 'required|string|max:5000',
    'content_html' => 'nullable|string|max:10000',
    'status' => 'required|in:draft,posted',
    'category' => 'nullable|string|max:100',
    'platforms' => 'nullable|json',
    'files.*' => 'nullable|file|mimes:jpeg,jpg,png,gif,mp4,mov,avi|max:10240', // 10MB max
]
```

**Response (Success - 201):**

```json
{
    "success": true,
    "message": "Post created successfully",
    "data": {
        "id": 123,
        "user_id": 1,
        "content": "This is my post content",
        "content_html": "<p>This is my post content</p>",
        "status": "draft",
        "category": "General",
        "platforms": ["facebook", "twitter"],
        "published_at": null,
        "created_at": "2024-01-15T10:30:00.000000Z",
        "updated_at": "2024-01-15T10:30:00.000000Z",
        "media": [
            {
                "id": 1,
                "post_id": 123,
                "user_id": 1,
                "file_name": "image1.jpg",
                "file_path": "posts/123/image1.jpg",
                "file_url": "http://your-domain.com/storage/posts/123/image1.jpg",
                "file_type": "image/jpeg",
                "file_size": 245678,
                "mime_type": "image/jpeg",
                "display_order": 0,
                "created_at": "2024-01-15T10:30:00.000000Z"
            }
        ]
    }
}
```

**Response (Error - 400):**

```json
{
    "success": false,
    "error": "Validation failed",
    "errors": {
        "content": ["The content field is required."],
        "status": ["The status field is required."]
    }
}
```

**Laravel Controller Implementation:**

```php
public function store(Request $request)
{
    $validated = $request->validate([
        'user_id' => 'required|exists:users,id',
        'content' => 'required|string|max:5000',
        'content_html' => 'nullable|string|max:10000',
        'status' => 'required|in:draft,posted',
        'category' => 'nullable|string|max:100',
        'platforms' => 'nullable|json',
        'files.*' => 'nullable|file|mimes:jpeg,jpg,png,gif,mp4,mov,avi|max:10240',
    ]);

    // Create the post
    $post = Post::create([
        'user_id' => $validated['user_id'],
        'content' => $validated['content'],
        'content_html' => $validated['content_html'] ?? null,
        'status' => $validated['status'], // 'draft' or 'posted'
        'category' => $validated['category'] ?? 'General',
        'platforms' => $validated['platforms'] ? json_decode($validated['platforms'], true) : null,
        'published_at' => $validated['status'] === 'posted' ? now() : null,
    ]);

    // Handle file uploads
    if ($request->hasFile('files')) {
        foreach ($request->file('files') as $index => $file) {
            $path = $file->store("posts/{$post->id}", 'public');

            PostMedia::create([
                'post_id' => $post->id,
                'user_id' => $post->user_id,
                'file_name' => $file->getClientOriginalName(),
                'file_path' => $path,
                'file_url' => Storage::url($path),
                'file_type' => $file->getMimeType(),
                'file_size' => $file->getSize(),
                'mime_type' => $file->getMimeType(),
                'display_order' => $index,
            ]);
        }
    }

    $post->load('media');

    return response()->json([
        'success' => true,
        'message' => 'Post created successfully',
        'data' => $post,
    ], 201);
}
```

---

### 2. Get All Posts

**Endpoint:** `GET /api/posts`

**Description:** Retrieves all posts for the authenticated user with optional filtering.

**Query Parameters:**

-   `status` (optional): Filter by status (`draft` or `posted`)
-   `category` (optional): Filter by category
-   `page` (optional): Page number (default: 1)
-   `per_page` (optional): Items per page (default: 20, max: 100)
-   `sort` (optional): Sort field (`created_at`, `updated_at`, `published_at`)
-   `order` (optional): Sort order (`asc` or `desc`)

**Example Request:**

```
GET /api/posts?status=draft&page=1&per_page=20&sort=updated_at&order=desc
```

**Response (Success - 200):**

```json
{
    "success": true,
    "data": [
        {
            "id": 123,
            "user_id": 1,
            "content": "This is a draft post",
            "content_html": "<p>This is a draft post</p>",
            "status": "draft",
            "category": "General",
            "platforms": ["facebook"],
            "published_at": null,
            "created_at": "2024-01-15T10:30:00.000000Z",
            "updated_at": "2024-01-15T11:45:00.000000Z",
            "media": [],
            "media_count": 0
        }
    ],
    "meta": {
        "current_page": 1,
        "per_page": 20,
        "total": 45,
        "last_page": 3,
        "from": 1,
        "to": 20
    }
}
```

**Laravel Controller Implementation:**

```php
public function index(Request $request)
{
    $user = $request->user();

    $query = Post::where('user_id', $user->id)
        ->with('media')
        ->withCount('media');

    // Filter by status
    if ($request->has('status')) {
        $query->where('status', $request->status);
    }

    // Filter by category
    if ($request->has('category')) {
        $query->where('category', $request->category);
    }

    // Sorting
    $sort = $request->get('sort', 'created_at');
    $order = $request->get('order', 'desc');
    $query->orderBy($sort, $order);

    // Pagination
    $perPage = min($request->get('per_page', 20), 100);
    $posts = $query->paginate($perPage);

    return response()->json([
        'success' => true,
        'data' => $posts->items(),
        'meta' => [
            'current_page' => $posts->currentPage(),
            'per_page' => $posts->perPage(),
            'total' => $posts->total(),
            'last_page' => $posts->lastPage(),
            'from' => $posts->firstItem(),
            'to' => $posts->lastItem(),
        ],
    ]);
}
```

---

### 3. Get Single Post

**Endpoint:** `GET /api/posts/{id}`

**Description:** Retrieves a single post by ID with all associated media.

**Response (Success - 200):**

```json
{
    "success": true,
    "data": {
        "id": 123,
        "user_id": 1,
        "content": "This is my post content",
        "content_html": "<p>This is my post content</p>",
        "status": "draft",
        "category": "General",
        "platforms": ["facebook", "twitter"],
        "published_at": null,
        "created_at": "2024-01-15T10:30:00.000000Z",
        "updated_at": "2024-01-15T10:30:00.000000Z",
        "media": [
            {
                "id": 1,
                "post_id": 123,
                "user_id": 1,
                "file_name": "image1.jpg",
                "file_path": "posts/123/image1.jpg",
                "file_url": "http://your-domain.com/storage/posts/123/image1.jpg",
                "file_type": "image/jpeg",
                "file_size": 245678,
                "mime_type": "image/jpeg",
                "display_order": 0,
                "created_at": "2024-01-15T10:30:00.000000Z"
            }
        ]
    }
}
```

**Laravel Controller Implementation:**

```php
public function show($id)
{
    $user = $request->user();

    $post = Post::where('user_id', $user->id)
        ->with('media')
        ->findOrFail($id);

    return response()->json([
        'success' => true,
        'data' => $post,
    ]);
}
```

---

### 4. Update Post (Draft Only)

**Endpoint:** `PUT /api/posts/{id}`

**Description:** Updates a post. **Only allowed if status is 'draft'**. The `status` field in the payload indicates the action.

**Request Body (JSON):**

```json
{
    "content": "Updated post content",
    "content_html": "<p>Updated post content</p>",
    "status": "draft",
    "category": "Marketing",
    "platforms": ["facebook", "twitter", "linkedin"]
}
```

**Request Body (Form Data - with new media):**

```
content: "Updated post content"
content_html: "<p>Updated post content</p>"
status: draft
category: Marketing
platforms: ["facebook", "twitter"]
files[]: [New File 1]
files[]: [New File 2]
remove_media_ids: [1, 2]
```

**Important:**

-   The `status` field should be `"draft"` when updating (user clicked "Save Draft")
-   If `status` is `"posted"`, you can optionally publish directly, but the recommended flow is to use the `/publish` endpoint

**Validation Rules:**

```php
[
    'content' => 'sometimes|required|string|max:5000',
    'content_html' => 'nullable|string|max:10000',
    'status' => 'sometimes|required|in:draft,posted',
    'category' => 'nullable|string|max:100',
    'platforms' => 'nullable|json',
    'files.*' => 'nullable|file|mimes:jpeg,jpg,png,gif,mp4,mov,avi|max:10240',
    'remove_media_ids' => 'nullable|array',
    'remove_media_ids.*' => 'exists:post_media,id',
]
```

**Response (Success - 200):**

```json
{
  "success": true,
  "message": "Post updated successfully",
  "data": {
    "id": 123,
    "user_id": 1,
    "content": "Updated post content",
    "content_html": "<p>Updated post content</p>",
    "status": "draft",
    "category": "Marketing",
    "platforms": ["facebook", "twitter", "linkedin"],
    "published_at": null,
    "created_at": "2024-01-15T10:30:00.000000Z",
    "updated_at": "2024-01-15T12:00:00.000000Z",
    "media": [...]
  }
}
```

**Response (Error - 400):**

```json
{
    "success": false,
    "error": "Cannot update post with status 'posted'. Only draft posts can be updated."
}
```

**Laravel Controller Implementation:**

```php
public function update(Request $request, $id)
{
    $user = $request->user();

    $post = Post::where('user_id', $user->id)->findOrFail($id);

    // Check if post is draft (only drafts can be updated)
    if ($post->status !== 'draft') {
        return response()->json([
            'success' => false,
            'error' => "Cannot update post with status 'posted'. Only draft posts can be updated.",
        ], 400);
    }

    $validated = $request->validate([
        'content' => 'sometimes|required|string|max:5000',
        'content_html' => 'nullable|string|max:10000',
        'status' => 'sometimes|required|in:draft,posted',
        'category' => 'nullable|string|max:100',
        'platforms' => 'nullable|json',
        'files.*' => 'nullable|file|mimes:jpeg,jpg,png,gif,mp4,mov,avi|max:10240',
        'remove_media_ids' => 'nullable|array',
        'remove_media_ids.*' => 'exists:post_media,id',
    ]);

    // Update post fields
    if (isset($validated['content'])) {
        $post->content = $validated['content'];
    }
    if (isset($validated['content_html'])) {
        $post->content_html = $validated['content_html'];
    }
    if (isset($validated['status'])) {
        $post->status = $validated['status'];
        // If status is 'posted', set published_at
        if ($validated['status'] === 'posted') {
            $post->published_at = now();
        }
    }
    if (isset($validated['category'])) {
        $post->category = $validated['category'];
    }
    if (isset($validated['platforms'])) {
        $post->platforms = json_decode($validated['platforms'], true);
    }

    $post->save();

    // Remove media if specified
    if (isset($validated['remove_media_ids'])) {
        foreach ($validated['remove_media_ids'] as $mediaId) {
            $media = PostMedia::find($mediaId);
            if ($media && $media->post_id === $post->id) {
                Storage::disk('public')->delete($media->file_path);
                $media->delete();
            }
        }
    }

    // Add new media files
    if ($request->hasFile('files')) {
        $existingMediaCount = $post->media()->count();
        foreach ($request->file('files') as $index => $file) {
            $path = $file->store("posts/{$post->id}", 'public');

            PostMedia::create([
                'post_id' => $post->id,
                'user_id' => $post->user_id,
                'file_name' => $file->getClientOriginalName(),
                'file_path' => $path,
                'file_url' => Storage::url($path),
                'file_type' => $file->getMimeType(),
                'file_size' => $file->getSize(),
                'mime_type' => $file->getMimeType(),
                'display_order' => $existingMediaCount + $index,
            ]);
        }
    }

    $post->load('media');

    return response()->json([
        'success' => true,
        'message' => 'Post updated successfully',
        'data' => $post,
    ]);
}
```

---

### 5. Publish Post

**Endpoint:** `PUT /api/posts/{id}/publish`

**Description:** Changes post status from 'draft' to 'posted' and sets published_at timestamp. **This endpoint is ONLY called when user clicks "Publish"**.

**Request Body (optional):**

```json
{
    "platforms": ["facebook", "twitter"]
}
```

**Response (Success - 200):**

```json
{
  "success": true,
  "message": "Post published successfully",
  "data": {
    "id": 123,
    "user_id": 1,
    "content": "This is my post content",
    "content_html": "<p>This is my post content</p>",
    "status": "posted",
    "category": "General",
    "platforms": ["facebook", "twitter"],
    "published_at": "2024-01-15T12:30:00.000000Z",
    "created_at": "2024-01-15T10:30:00.000000Z",
    "updated_at": "2024-01-15T12:30:00.000000Z",
    "media": [...]
  }
}
```

**Response (Error - 400):**

```json
{
    "success": false,
    "error": "Post is already published"
}
```

**Laravel Controller Implementation:**

```php
public function publish(Request $request, $id)
{
    $user = $request->user();

    $post = Post::where('user_id', $user->id)->findOrFail($id);

    // Check if post is already published
    if ($post->status === 'posted') {
        return response()->json([
            'success' => false,
            'error' => 'Post is already published',
        ], 400);
    }

    // Update platforms if provided
    if ($request->has('platforms')) {
        $post->platforms = $request->platforms;
    }

    // Change status to posted and set published_at
    $post->status = 'posted';
    $post->published_at = now();
    $post->save();

    $post->load('media');

    return response()->json([
        'success' => true,
        'message' => 'Post published successfully',
        'data' => $post,
    ]);
}
```

---

### 6. Delete Post

**Endpoint:** `DELETE /api/posts/{id}`

**Description:** Deletes a post and all associated media files. Works for both drafts and published posts.

**Response (Success - 200):**

```json
{
    "success": true,
    "message": "Post deleted successfully"
}
```

**Laravel Controller Implementation:**

```php
public function destroy($id)
{
    $user = $request->user();

    $post = Post::where('user_id', $user->id)->findOrFail($id);

    // Delete all media files
    foreach ($post->media as $media) {
        Storage::disk('public')->delete($media->file_path);
        $media->delete();
    }

    // Delete the post
    $post->delete();

    return response()->json([
        'success' => true,
        'message' => 'Post deleted successfully',
    ]);
}
```

---

### 7. Upload Media to Post

**Endpoint:** `POST /api/posts/{id}/media`

**Description:** Adds media files to an existing post. Only allowed for draft posts.

**Request Body (Form Data):**

```
files[]: [File 1]
files[]: [File 2]
display_order: 0
```

**Response (Success - 201):**

```json
{
    "success": true,
    "message": "Media uploaded successfully",
    "data": [
        {
            "id": 3,
            "post_id": 123,
            "user_id": 1,
            "file_name": "new_image.jpg",
            "file_path": "posts/123/new_image.jpg",
            "file_url": "http://your-domain.com/storage/posts/123/new_image.jpg",
            "file_type": "image/jpeg",
            "file_size": 345678,
            "mime_type": "image/jpeg",
            "display_order": 0,
            "created_at": "2024-01-15T12:00:00.000000Z"
        }
    ]
}
```

---

### 8. Delete Media from Post

**Endpoint:** `DELETE /api/posts/{id}/media/{media_id}`

**Description:** Removes a media file from a post. Only allowed for draft posts.

**Response (Success - 200):**

```json
{
    "success": true,
    "message": "Media deleted successfully"
}
```

---

## 🗄️ Database Schema

### Posts Table Migration

```php
Schema::create('posts', function (Blueprint $table) {
    $table->id();
    $table->foreignId('user_id')->constrained()->onDelete('cascade');
    $table->text('content');
    $table->text('content_html')->nullable();
    $table->enum('status', ['draft', 'posted'])->default('draft');
    $table->string('category', 100)->nullable()->default('General');
    $table->json('platforms')->nullable();
    $table->timestamp('published_at')->nullable();
    $table->timestamps();

    $table->index('user_id');
    $table->index('status');
    $table->index(['user_id', 'status']);
    $table->index('published_at');
    $table->index('created_at');
});
```

### Post Media Table Migration

```php
Schema::create('post_media', function (Blueprint $table) {
    $table->id();
    $table->foreignId('post_id')->constrained()->onDelete('cascade');
    $table->foreignId('user_id')->constrained()->onDelete('cascade');
    $table->string('file_name', 255);
    $table->string('file_path', 500);
    $table->string('file_url', 500)->nullable();
    $table->string('file_type', 50)->nullable();
    $table->bigInteger('file_size')->nullable();
    $table->string('mime_type', 100)->nullable();
    $table->unsignedInteger('display_order')->default(0);
    $table->timestamps();

    $table->index('post_id');
    $table->index('user_id');
    $table->index(['post_id', 'display_order']);
});
```

---

## 🔐 Route Definition

```php
// routes/api.php
Route::middleware('auth:sanctum')->group(function () {
    // Posts CRUD
    Route::apiResource('posts', PostController::class);

    // Publish endpoint
    Route::put('posts/{id}/publish', [PostController::class, 'publish']);

    // Media endpoints
    Route::post('posts/{id}/media', [PostMediaController::class, 'store']);
    Route::delete('posts/{id}/media/{media_id}', [PostMediaController::class, 'destroy']);
});
```

---

## 📊 Summary: How to Identify Draft vs Publish

### When User Clicks "Save Draft":

1. Frontend sends: `POST /api/posts` or `PUT /api/posts/{id}` with `status: "draft"`
2. Backend receives: `status === "draft"` in the payload
3. Backend action: Save post with `status = 'draft'`

### When User Clicks "Publish":

1. Frontend first sends: `POST /api/posts` or `PUT /api/posts/{id}` with `status: "draft"` (if not already saved)
2. Frontend then sends: `PUT /api/posts/{id}/publish`
3. Backend action: Change `status` from `'draft'` to `'posted'` and set `published_at`

### Key Points:

-   ✅ Always check the `status` field in create/update requests
-   ✅ The `/publish` endpoint is ONLY called for publish action
-   ✅ Only draft posts can be updated
-   ✅ Published posts are read-only

---

## ✅ Testing Checklist

-   [ ] Create draft post (status: "draft")
-   [ ] Create draft post with media files
-   [ ] Update draft post
-   [ ] Publish draft post
-   [ ] Try to update published post (should fail)
-   [ ] Delete post
-   [ ] Get all posts with filters
-   [ ] Upload media to draft post
-   [ ] Delete media from draft post
-   [ ] Test authentication (should fail without token)
-   [ ] Test authorization (user can only access own posts)

---

## 🚨 Common Issues & Solutions

### Issue 1: Status field not being sent

**Solution:** Ensure frontend always includes `status: "draft"` in create/update requests

### Issue 2: Can't distinguish draft vs publish

**Solution:** Check the `status` field in the payload. If it's missing, default to `"draft"`

### Issue 3: Published posts can be edited

**Solution:** Add validation in `update()` method to check `status === 'draft'`

### Issue 4: Media files not uploading

**Solution:** Check file size limits, MIME types, and storage configuration

---

**Last Updated:** 2024-01-15
**Version:** 1.0.0
