# Laravel API Endpoints - Quick Reference

## Base URL

```
http://social-platform/api
```

All endpoints require authentication: `Authorization: Bearer {token}`

---

## 📝 Posts Endpoints

### 1. Create Post (Draft)

```
POST /api/posts
```

**Body:** `multipart/form-data` or `application/json`

-   `user_id` (required)
-   `content` (required)
-   `content_html` (optional)
-   `status` (default: "draft")
-   `category` (optional)
-   `platforms` (optional, JSON array)
-   `files[]` (optional, for media upload)

**Response:** `201 Created` - Post object with media

---

### 2. Get All Posts

```
GET /api/posts
```

**Query Parameters:**

-   `status` (optional): `draft` | `posted`
-   `category` (optional)
-   `page` (optional, default: 1)
-   `per_page` (optional, default: 20)
-   `sort` (optional): `created_at` | `updated_at` | `published_at`
-   `order` (optional): `asc` | `desc`

**Response:** `200 OK` - Paginated list of posts

**Example:**

```
GET /api/posts?status=draft&page=1&per_page=20&sort=updated_at&order=desc
```

---

### 3. Get Single Post

```
GET /api/posts/{id}
```

**Response:** `200 OK` - Post object with media

---

### 4. Update Post (Draft Only)

```
PUT /api/posts/{id}
```

**Body:** `multipart/form-data` or `application/json`

-   `content` (optional)
-   `content_html` (optional)
-   `category` (optional)
-   `platforms` (optional)
-   `files[]` (optional, new media)
-   `remove_media_ids` (optional, array of media IDs to delete)

**Response:** `200 OK` - Updated post object

**Note:** Only works if `status = 'draft'`

---

### 5. Publish Post

```
PUT /api/posts/{id}/publish
```

**Body (optional):**

```json
{
    "platforms": ["facebook", "twitter"]
}
```

**Response:** `200 OK` - Post object with `status = 'posted'` and `published_at` set

**Note:** Changes status from `draft` to `posted`

---

### 6. Delete Post

```
DELETE /api/posts/{id}
```

**Response:** `200 OK` - Success message

**Note:** Deletes post and all associated media files

---

## 🖼️ Media Endpoints

### 7. Upload Media to Post

```
POST /api/posts/{id}/media
```

**Body:** `multipart/form-data`

-   `files[]` (required, one or more files)
-   `display_order` (optional, starting order number)

**Response:** `201 Created` - Array of media objects

**Note:** Only works if post `status = 'draft'`

---

### 8. Delete Media from Post

```
DELETE /api/posts/{id}/media/{media_id}
```

**Response:** `200 OK` - Success message

**Note:** Only works if post `status = 'draft'`

---

## 📊 Response Format

### Success Response

```json
{
  "success": true,
  "message": "Operation successful",
  "data": { ... }
}
```

### Error Response

```json
{
    "success": false,
    "error": "Error message",
    "errors": {
        "field": ["Validation error"]
    }
}
```

---

## 🔒 Status Rules

| Action       | Draft Post | Posted Post            |
| ------------ | ---------- | ---------------------- |
| View         | ✅         | ✅                     |
| Edit         | ✅         | ❌                     |
| Delete       | ✅         | ✅                     |
| Publish      | ✅         | ❌ (already published) |
| Add Media    | ✅         | ❌                     |
| Remove Media | ✅         | ❌                     |

---

## 📋 Database Schema Summary

### `posts` Table

-   `id` - Primary key
-   `user_id` - Foreign key to users
-   `content` - Post text content
-   `content_html` - HTML formatted content
-   `status` - `'draft'` or `'posted'`
-   `category` - Post category
-   `platforms` - JSON array
-   `published_at` - Timestamp (NULL for drafts)
-   `created_at` - Creation timestamp
-   `updated_at` - Last update timestamp

### `post_media` Table

-   `id` - Primary key
-   `post_id` - Foreign key to posts
-   `user_id` - Foreign key to users
-   `file_name` - Original filename
-   `file_path` - Storage path
-   `file_url` - Public URL
-   `file_type` - MIME type
-   `file_size` - Size in bytes
-   `display_order` - Display order
-   `created_at` - Creation timestamp

---

## 🚀 Quick Implementation Guide

### Laravel Route Definition

```php
// routes/api.php
Route::middleware('auth:sanctum')->group(function () {
    // Posts
    Route::apiResource('posts', PostController::class);
    Route::put('posts/{id}/publish', [PostController::class, 'publish']);

    // Media
    Route::post('posts/{id}/media', [PostMediaController::class, 'store']);
    Route::delete('posts/{id}/media/{media_id}', [PostMediaController::class, 'destroy']);
});
```

### Controller Method Example (Store)

```php
public function store(Request $request)
{
    $validated = $request->validate([
        'user_id' => 'required|exists:users,id',
        'content' => 'required|string',
        'content_html' => 'nullable|string',
        'status' => 'in:draft,posted',
        'category' => 'nullable|string|max:100',
        'platforms' => 'nullable|json',
        'files.*' => 'nullable|file|max:10240', // 10MB max
    ]);

    $post = Post::create([
        'user_id' => $validated['user_id'],
        'content' => $validated['content'],
        'content_html' => $validated['content_html'] ?? null,
        'status' => $validated['status'] ?? 'draft',
        'category' => $validated['category'] ?? 'General',
        'platforms' => $validated['platforms'] ? json_decode($validated['platforms']) : null,
    ]);

    // Handle file uploads
    if ($request->hasFile('files')) {
        foreach ($request->file('files') as $index => $file) {
            $path = $file->store("posts/{$post->id}", 'public');
            PostMedia::create([
                'post_id' => $post->id,
                'user_id' => $post->user_id,
                'file_name' => $file->getClientOriginalName(),
                'file_path' => $path,
                'file_url' => Storage::url($path),
                'file_type' => $file->getMimeType(),
                'file_size' => $file->getSize(),
                'display_order' => $index,
            ]);
        }
    }

    $post->load('media');

    return response()->json([
        'success' => true,
        'message' => 'Post created successfully',
        'data' => $post,
    ], 201);
}
```

---

## 🔍 Common Use Cases

### 1. Create Draft with Images

```bash
POST /api/posts
Content-Type: multipart/form-data

user_id=1
content=My post content
category=Marketing
files[]=@image1.jpg
files[]=@image2.jpg
```

### 2. Get All Drafts

```bash
GET /api/posts?status=draft&sort=updated_at&order=desc
```

### 3. Publish a Draft

```bash
PUT /api/posts/123/publish
Content-Type: application/json

{
  "platforms": ["facebook", "twitter"]
}
```

### 4. Update Draft Content

```bash
PUT /api/posts/123
Content-Type: application/json

{
  "content": "Updated content",
  "category": "News"
}
```

### 5. Add More Images to Draft

```bash
POST /api/posts/123/media
Content-Type: multipart/form-data

files[]=@new_image.jpg
```

---

## ⚠️ Important Notes

1. **Authentication Required**: All endpoints require valid Bearer token
2. **Status Validation**: Only draft posts can be edited/updated
3. **User Authorization**: Users can only access their own posts
4. **File Storage**: Configure Laravel storage (local or S3)
5. **Media Cleanup**: Delete media files when post/media is deleted
6. **JSON Fields**: `platforms` should be stored as JSON in database
7. **Pagination**: List endpoints support pagination (default: 20 per page)

---

For detailed documentation, see `POST_DRAFT_API_DOCUMENTATION.md`
