# Improved Password Reset Workflow - Security Enhanced

## Overview

This document describes the **improved and more secure** password reset workflow that has been implemented. The key security improvement is that **only the token is passed in the URL**, not the email address. This prevents email exposure in browser history, server logs, and other places.

## Security Improvements

### Before (Old Implementation)

-   URL contained both token and email: `/reset-password?token=xxx&email=user@example.com`
-   Email was exposed in URL, browser history, server logs
-   Frontend had to send email in request body

### After (New Implementation)

-   URL contains only token: `/reset-password?token=xxx`
-   Email is retrieved from token record in database (more secure)
-   Frontend validates token before showing form
-   Email is never exposed in URLs or request bodies

## Complete Workflow

### Step 1: User Requests Password Reset

**Endpoint**: `POST /api/forgot-password`

**Request**:

```json
{
    "email": "user@example.com"
}
```

**Response**:

```json
{
    "success": true,
    "message": "If an account exists with that email, we have sent a password reset link."
}
```

**What Happens**:

1. Backend generates a secure 64-character random token
2. Token is hashed and stored in `password_reset_tokens` table with email
3. Email is sent with reset link: `http://frontend.com/reset-password?token=TOKEN_HERE`
4. **Note**: Only token is in URL, not email!

---

### Step 2: User Clicks Reset Link

**What Happens**:

1. User clicks link from email
2. Browser navigates to: `http://frontend.com/reset-password?token=abc123...`
3. Frontend page loads and **immediately** calls validation endpoint

---

### Step 3: Frontend Validates Token (NEW ENDPOINT)

**Endpoint**: `GET /api/validate-reset-token?token=xxx`

**Purpose**: Check if token is valid and not expired **before** showing password form

**Response (Valid Token)**:

```json
{
    "success": true,
    "message": "Reset token is valid.",
    "valid": true,
    "expired": false,
    "email": "u***@example.com",
    "expires_in_minutes": 45
}
```

**Response (Invalid Token)**:

```json
{
    "success": false,
    "message": "Invalid or expired reset token.",
    "errors": {
        "token": ["This password reset token is invalid or has expired."]
    },
    "valid": false,
    "expired": false
}
```

**Response (Expired Token)**:

```json
{
    "success": false,
    "message": "This password reset token has expired. Please request a new one.",
    "errors": {
        "token": [
            "This password reset token has expired. Please request a new password reset link."
        ]
    },
    "valid": false,
    "expired": true
}
```

**Frontend Action**:

-   If `valid: true` → Show password reset form
-   If `valid: false` and `expired: true` → Show "Token expired" message with link to request new reset
-   If `valid: false` and `expired: false` → Show "Invalid token" error

---

### Step 4: User Submits New Password

**Endpoint**: `POST /api/reset-password`

**Request**:

```json
{
    "token": "abc123...",
    "password": "newpassword123",
    "password_confirmation": "newpassword123"
}
```

**Note**: Email is **NOT** required in request! Backend retrieves email from token record.

**Response (Success)**:

```json
{
    "success": true,
    "message": "Your password has been reset successfully. You can now login with your new password."
}
```

**Response (Error - Invalid/Expired Token)**:

```json
{
    "success": false,
    "message": "Invalid or expired reset token.",
    "errors": {
        "token": ["This password reset token is invalid or has expired."]
    }
}
```

**What Happens**:

1. Backend finds token record (searches hashed tokens)
2. Retrieves email from token record
3. Validates token is not expired (60 minutes)
4. Updates user password
5. Deletes used token
6. Returns success

---

## Frontend Implementation Example (Next.js)

### Reset Password Page Component

```javascript
import { useRouter } from "next/router";
import { useState, useEffect } from "react";
import axios from "axios";

export default function ResetPassword() {
    const router = useRouter();
    const { token } = router.query; // Only token, no email!

    const [password, setPassword] = useState("");
    const [passwordConfirmation, setPasswordConfirmation] = useState("");
    const [loading, setLoading] = useState(false);
    const [validating, setValidating] = useState(true);
    const [tokenValid, setTokenValid] = useState(false);
    const [tokenExpired, setTokenExpired] = useState(false);
    const [message, setMessage] = useState("");
    const [errors, setErrors] = useState({});
    const [maskedEmail, setMaskedEmail] = useState("");

    // Validate token when page loads
    useEffect(() => {
        if (!token) {
            setValidating(false);
            setTokenValid(false);
            setMessage(
                "Invalid reset link. Please request a new password reset."
            );
            return;
        }

        // Call validation endpoint
        validateToken();
    }, [token]);

    const validateToken = async () => {
        try {
            const response = await axios.get(
                `${process.env.NEXT_PUBLIC_API_URL}/api/validate-reset-token`,
                {
                    params: { token },
                }
            );

            if (response.data.success && response.data.valid) {
                setTokenValid(true);
                setMaskedEmail(response.data.email || "");
                setMessage(
                    `Reset token is valid. Please enter your new password.`
                );
            } else {
                setTokenValid(false);
                setTokenExpired(response.data.expired || false);
                setMessage(
                    response.data.message || "Invalid or expired token."
                );
            }
        } catch (error) {
            const errorData = error.response?.data;
            setTokenValid(false);
            setTokenExpired(errorData?.expired || false);
            setMessage(errorData?.message || "Invalid or expired reset token.");
        } finally {
            setValidating(false);
        }
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        setLoading(true);
        setErrors({});
        setMessage("");

        try {
            const response = await axios.post(
                `${process.env.NEXT_PUBLIC_API_URL}/api/reset-password`,
                {
                    token,
                    password,
                    password_confirmation: passwordConfirmation,
                }
            );

            if (response.data.success) {
                setMessage(
                    "Password reset successful! Redirecting to login..."
                );
                // Redirect to login page after 2 seconds
                setTimeout(() => {
                    router.push("/login");
                }, 2000);
            }
        } catch (error) {
            const errorData = error.response?.data;
            setMessage(errorData?.message || "Failed to reset password.");
            setErrors(errorData?.errors || {});
        } finally {
            setLoading(false);
        }
    };

    // Show loading state while validating
    if (validating) {
        return (
            <div className="container">
                <h1>Validating reset token...</h1>
                <p>Please wait while we verify your reset link.</p>
            </div>
        );
    }

    // Show error if token is invalid or expired
    if (!tokenValid) {
        return (
            <div className="container">
                <h1>Invalid or Expired Reset Link</h1>
                <p>{message}</p>
                {tokenExpired && (
                    <p>
                        <a href="/forgot-password">
                            Request a new password reset link
                        </a>
                    </p>
                )}
            </div>
        );
    }

    // Show password reset form
    return (
        <div className="container">
            <h1>Reset Your Password</h1>
            {maskedEmail && <p>Resetting password for: {maskedEmail}</p>}

            {message && (
                <div className={`alert ${errors.token ? "error" : "success"}`}>
                    {message}
                </div>
            )}

            <form onSubmit={handleSubmit}>
                <div>
                    <label>New Password</label>
                    <input
                        type="password"
                        value={password}
                        onChange={(e) => setPassword(e.target.value)}
                        required
                        minLength={6}
                    />
                    {errors.password && (
                        <span className="error">{errors.password[0]}</span>
                    )}
                </div>

                <div>
                    <label>Confirm Password</label>
                    <input
                        type="password"
                        value={passwordConfirmation}
                        onChange={(e) =>
                            setPasswordConfirmation(e.target.value)
                        }
                        required
                        minLength={6}
                    />
                    {errors.password_confirmation && (
                        <span className="error">
                            {errors.password_confirmation[0]}
                        </span>
                    )}
                </div>

                <button type="submit" disabled={loading}>
                    {loading ? "Resetting..." : "Reset Password"}
                </button>
            </form>
        </div>
    );
}
```

---

## Key Benefits

1. **Enhanced Security**: Email is never exposed in URLs or request bodies
2. **Better UX**: Frontend can validate token immediately and show appropriate messages
3. **Token Expiration Handling**: Users are informed immediately if token is expired
4. **Privacy**: Masked email shown to user (e.g., `u***@example.com`)
5. **Performance**: Only checks tokens from last 24 hours (optimization)

---

## API Endpoints Summary

| Method | Endpoint                    | Purpose                                | Request Body                                 |
| ------ | --------------------------- | -------------------------------------- | -------------------------------------------- |
| POST   | `/api/forgot-password`      | Request password reset                 | `{ email }`                                  |
| GET    | `/api/validate-reset-token` | Validate token (when user clicks link) | Query param: `token`                         |
| POST   | `/api/reset-password`       | Reset password                         | `{ token, password, password_confirmation }` |

---

## Token Expiration

-   **Token Lifetime**: 60 minutes
-   **Validation**: Checked both when validating token and when resetting password
-   **Cleanup**: Expired tokens are automatically deleted

---

## Security Best Practices Implemented

1. ✅ Tokens are hashed in database
2. ✅ Only token in URL (no email exposure)
3. ✅ Email retrieved from database, not request
4. ✅ Token expiration checked at multiple points
5. ✅ Email enumeration prevention (same response for existing/non-existing emails)
6. ✅ Token validation endpoint for immediate feedback
7. ✅ Optimized token lookup (only checks recent tokens)

---

## Testing

### Test Valid Token Flow

1. Request password reset: `POST /api/forgot-password` with email
2. Get token from debug response (development only)
3. Validate token: `GET /api/validate-reset-token?token=xxx`
4. Should return `valid: true`
5. Reset password: `POST /api/reset-password` with token and password
6. Should succeed

### Test Expired Token Flow

1. Request password reset
2. Wait 61+ minutes (or manually update `created_at` in database)
3. Validate token: `GET /api/validate-reset-token?token=xxx`
4. Should return `expired: true`
5. Frontend should show "Token expired" message

### Test Invalid Token Flow

1. Use invalid token: `GET /api/validate-reset-token?token=invalid`
2. Should return `valid: false, expired: false`
3. Frontend should show "Invalid token" message

---

## Migration Notes

If you have existing reset links with email in URL, they will no longer work. Users will need to request new reset links. This is expected behavior for the security improvement.
