# How to Run Scheduled Posts - Complete Guide

## 🎯 How It Works

The scheduled posts system works in **3 layers**:

```
1. Laravel Scheduler (runs every minute)
   ↓
2. Console Command (posts:publish-scheduled)
   ↓
3. Background Job (PublishScheduledPosts)
   ↓
4. Publishes posts where scheduled_at <= NOW()
```

## 📋 Two Ways to Run

### **Option 1: With Queue System (Recommended for Production)**

-   Job runs in background queue
-   Non-blocking, better performance
-   Can handle multiple jobs simultaneously
-   Requires queue worker running

### **Option 2: Without Queue (Synchronous - Simpler for Development)**

-   Job runs immediately when command executes
-   Simpler setup, no queue worker needed
-   Blocks until completion
-   Good for testing

---

## 🚀 Setup Instructions

### **Step 1: Run Database Migration**

First, make sure the migration is applied:

```bash
php artisan migrate
```

This will:

-   Add `scheduled_at` column to `posts` table
-   Update `status` enum to include `'scheduled'`
-   Create necessary indexes

---

### **Step 2: Choose Your Approach**

## **Approach A: With Queue System (Production)**

### A1. Ensure Jobs Table Exists

The jobs table should already exist (Laravel creates it by default). If not:

```bash
php artisan queue:table
php artisan migrate
```

### A2. Start Queue Worker

**For Development:**

```bash
# In a separate terminal window
php artisan queue:work
```

**For Production (using Supervisor):**
Create `/etc/supervisor/conf.d/laravel-worker.conf`:

```ini
[program:laravel-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path-to-your-project/artisan queue:work database --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=1
redirect_stderr=true
stdout_logfile=/path-to-your-project/storage/logs/worker.log
stopwaitsecs=3600
```

Then:

```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start laravel-worker:*
```

### A3. Set Up Laravel Scheduler (Cron Job)

**For Development (Manual Testing):**
You can run the scheduler manually:

```bash
php artisan schedule:run
```

**For Production (Automatic):**
Add to your server's crontab:

```bash
crontab -e
```

Add this line (runs every minute):

```bash
* * * * * cd /path-to-your-project && php artisan schedule:run >> /dev/null 2>&1
```

**For Windows (Development):**
Use Task Scheduler or run manually in a terminal:

```powershell
# Run this in PowerShell (keep it running)
while ($true) {
    php artisan schedule:run
    Start-Sleep -Seconds 60
}
```

---

## **Approach B: Without Queue (Synchronous - Simpler)**

If you want to run without queue system, we need to modify the command to run synchronously.

### B1. Update the Command

The command already dispatches the job. We can modify it to run synchronously by changing the job dispatch.

**Option B1a: Modify Command to Run Directly**

Update `app/Console/Commands/PublishScheduledPostsCommand.php`:

```php
public function handle(): int
{
    $this->info('Checking for scheduled posts to publish...');

    try {
        // Run synchronously instead of queuing
        $job = new \App\Jobs\PublishScheduledPosts();
        $job->handle();

        $this->info('Scheduled posts check completed successfully');
        return Command::SUCCESS;
    } catch (\Exception $e) {
        $this->error('Error checking scheduled posts: ' . $e->getMessage());
        return Command::FAILURE;
    }
}
```

**Option B1b: Use dispatchSync()**

Or keep the command as is, but change the job to use `dispatchSync()`:

In `app/Console/Commands/PublishScheduledPostsCommand.php`:

```php
use Illuminate\Support\Facades\Bus;

public function handle(): int
{
    $this->info('Checking for scheduled posts to publish...');

    try {
        // Dispatch synchronously (runs immediately, no queue)
        Bus::dispatchSync(new PublishScheduledPosts());

        $this->info('Scheduled posts check completed successfully');
        return Command::SUCCESS;
    } catch (\Exception $e) {
        $this->error('Error checking scheduled posts: ' . $e->getMessage());
        return Command::FAILURE;
    }
}
```

### B2. Set Up Laravel Scheduler (Same as Approach A)

Follow Step A3 above to set up the scheduler.

---

## 🧪 Testing the System

### **Test 1: Create a Scheduled Post**

Use your API or frontend to create a post scheduled for a few minutes in the future:

```bash
PUT /api/posts/1/publish
{
  "platforms": ["facebook"],
  "scheduled_at": "2025-01-23T15:05:00Z"  # 5 minutes from now
}
```

### **Test 2: Run Command Manually**

Test the command directly:

```bash
php artisan posts:publish-scheduled
```

You should see:

```
Checking for scheduled posts to publish...
Scheduled posts check completed successfully
```

### **Test 3: Check Logs**

Check `storage/logs/laravel.log` for:

```
PublishScheduledPosts: Found 1 scheduled post(s) to publish
PublishScheduledPosts: Published post ID 1
```

### **Test 4: Test with Past Date**

Create a post scheduled in the past to test immediate publishing:

```bash
# Update a post's scheduled_at to 1 minute ago
php artisan tinker
```

```php
$post = App\Models\Post::find(1);
$post->scheduled_at = now()->subMinute();
$post->status = 'scheduled';
$post->save();
exit
```

Then run:

```bash
php artisan posts:publish-scheduled
```

The post should be published immediately.

### **Test 5: Test Scheduler**

Run the scheduler manually:

```bash
php artisan schedule:run
```

This should execute the `posts:publish-scheduled` command if it's due.

---

## 🔍 Monitoring & Debugging

### **Check Scheduled Posts**

```bash
php artisan tinker
```

```php
// See all scheduled posts
App\Models\Post::where('status', 'scheduled')->get();

// See posts scheduled to publish soon
App\Models\Post::where('status', 'scheduled')
    ->where('scheduled_at', '<=', now()->addMinutes(5))
    ->get();
```

### **Check Queue Jobs**

If using queue system:

```bash
# See pending jobs
php artisan queue:work --once

# See failed jobs
php artisan queue:failed

# Retry failed jobs
php artisan queue:retry all
```

### **Check Logs**

```bash
# View logs
tail -f storage/logs/laravel.log

# Or use Laravel Pail (if installed)
php artisan pail
```

---

## ⚙️ Configuration Options

### **Change Check Frequency**

Currently set to run every minute. To change:

In `routes/console.php`:

```php
// Run every 5 minutes instead
Schedule::command('posts:publish-scheduled')
    ->everyFiveMinutes()
    ->withoutOverlapping()
    ->runInBackground();
```

### **Change Queue Connection**

In `.env`:

```env
QUEUE_CONNECTION=database  # or 'sync', 'redis', etc.
```

### **Run Without Background**

If you want the scheduler to wait for completion:

In `routes/console.php`:

```php
Schedule::command('posts:publish-scheduled')
    ->everyMinute()
    ->withoutOverlapping();
    // Remove ->runInBackground()
```

---

## 🎯 Recommended Setup by Environment

### **Development (Local)**

-   Use **Approach B** (Synchronous) - simpler, no queue worker needed
-   Run scheduler manually: `php artisan schedule:run`
-   Or use Task Scheduler (Windows) / cron (Linux/Mac)

### **Production**

-   Use **Approach A** (With Queue) - better performance
-   Set up Supervisor for queue worker
-   Set up cron for Laravel scheduler
-   Monitor logs regularly

---

## ✅ Quick Start Checklist

-   [ ] Run migration: `php artisan migrate`
-   [ ] Choose approach (A or B)
-   [ ] If Approach A: Start queue worker (`php artisan queue:work`)
-   [ ] Set up scheduler (cron or manual)
-   [ ] Test with a scheduled post
-   [ ] Check logs to verify it's working
-   [ ] Monitor for any errors

---

## 🚨 Troubleshooting

### **Problem: Posts not publishing**

1. **Check if scheduler is running:**

    ```bash
    php artisan schedule:list
    ```

2. **Check if queue worker is running (if using queue):**

    ```bash
    # Check if process is running
    ps aux | grep "queue:work"
    ```

3. **Check logs:**

    ```bash
    tail -f storage/logs/laravel.log
    ```

4. **Run command manually:**
    ```bash
    php artisan posts:publish-scheduled
    ```

### **Problem: Queue jobs not processing**

1. **Check queue connection in `.env`:**

    ```env
    QUEUE_CONNECTION=database
    ```

2. **Check if jobs table exists:**

    ```bash
    php artisan migrate
    ```

3. **Check failed jobs:**
    ```bash
    php artisan queue:failed
    ```

### **Problem: Scheduler not running**

1. **Check cron is set up correctly:**

    ```bash
    crontab -l
    ```

2. **Test scheduler manually:**

    ```bash
    php artisan schedule:run -v
    ```

3. **Check Laravel logs for scheduler errors**

---

## 📝 Summary

**The system works like this:**

1. **Laravel Scheduler** runs every minute (via cron)
2. **Scheduler** calls `posts:publish-scheduled` command
3. **Command** dispatches `PublishScheduledPosts` job
4. **Job** finds posts where `scheduled_at <= NOW()`
5. **Job** publishes those posts (status: `'scheduled'` → `'posted'`)

**To make it work:**

1. Run migration
2. Choose queue or sync approach
3. Set up scheduler (cron or manual)
4. If using queue: start queue worker
5. Test with a scheduled post
6. Monitor logs

That's it! Your scheduled posts will now publish automatically. 🎉
