# Email Verification and Account Status Logic

## Understanding the Two Fields

### 1. `status` Field
- **Purpose**: Account status/state (active, inactive, suspended)
- **Set on Registration**: Automatically set to `'active'` (from model defaults)
- **Who Controls**: Admin or system (can be changed by admin)
- **Values**: `'active'`, `'inactive'`, `'suspended'`
- **When Checked**: During login (must be `'active'` to login)

### 2. `email_verified_at` Field
- **Purpose**: Email verification timestamp
- **Set on Registration**: `null` (not verified yet)
- **Who Controls**: User (when they click verification link)
- **Values**: `null` (not verified) or `timestamp` (verified)
- **When Checked**: Optional during login (can require verification)

---

## Current Logic Flow

### Registration Flow
```
1. User submits registration form
   ↓
2. Validate: name, email, password
   ↓
3. Create AppUser with:
   - name: from request
   - email: from request
   - password: hashed
   - status: 'active' (default from model)
   - email_verified_at: null (not verified yet)
   - timezone: 'UTC' (default)
   ↓
4. Return success response
```

**Why `email_verified_at` is null?**
- ✅ **Correct behavior**: User hasn't verified their email yet
- Email verification is a separate step after registration
- User receives verification email and clicks link
- Only then `email_verified_at` is set to current timestamp

### Login Flow
```
1. User submits email and password
   ↓
2. Check: User exists and password matches
   ↓
3. Check: status === 'active'
   - If not active → Return 403 error
   ↓
4. Optional: Check email_verified_at is not null
   - Currently commented out (optional)
   - Uncomment if you want to require email verification
   ↓
5. Create API token
   ↓
6. Return token and user data
```

---

## Status vs Email Verification

### Scenario 1: New User (Just Registered)
```
status: 'active'
email_verified_at: null
```
- ✅ Can login (status is active)
- ⚠️ Email not verified yet
- **Action**: User can login, but you might want to show "Please verify email" message

### Scenario 2: Verified User
```
status: 'active'
email_verified_at: '2025-01-20 10:30:00'
```
- ✅ Can login
- ✅ Email verified
- **Action**: Full access

### Scenario 3: Suspended User
```
status: 'suspended'
email_verified_at: '2025-01-20 10:30:00'
```
- ❌ Cannot login (status is not 'active')
- ✅ Email verified (but doesn't matter, account is suspended)
- **Action**: Return 403 error

### Scenario 4: Inactive User
```
status: 'inactive'
email_verified_at: null
```
- ❌ Cannot login (status is not 'active')
- ❌ Email not verified
- **Action**: Return 403 error

---

## Implementation Options

### Option 1: Allow Login Without Email Verification (Current)
```php
// Login works if status === 'active'
// Email verification is optional
```
**Use Case**: Quick onboarding, verify email later

### Option 2: Require Email Verification (Strict)
```php
// Uncomment the email verification check in login method
if (!$user->email_verified_at) {
    return response()->json([
        'success' => false,
        'message' => 'Please verify your email address before logging in.',
    ], 403);
}
```
**Use Case**: Security-focused, must verify email first

### Option 3: Allow Login But Show Warning
```php
// Login succeeds, but return flag in response
'email_verified' => $user->email_verified_at !== null
```
**Use Case**: User can login, but frontend shows "Please verify email" banner

---

## Current Code Behavior

### Registration
```php
AppUser::create([
    'name' => $validated['name'],
    'email' => $validated['email'],
    'password' => Hash::make($validated['password']),
    // status: 'active' (from model default)
    // email_verified_at: null (not set, stays null)
]);
```

### Login Checks
```php
// ✅ Check 1: User exists and password correct
if (!$user || !Hash::check(...)) {
    return 401;
}

// ✅ Check 2: Status is active
if ($user->status !== 'active') {
    return 403; // Account not active
}

// ⚠️ Check 3: Email verified (currently optional/commented)
// Uncomment if you want to require email verification
```

---

## Recommended Approach

### For Most Applications:
1. **Allow login without email verification** (current behavior)
2. **Check status === 'active'** (required - already implemented)
3. **Return email verification status** in login response
4. **Frontend shows warning** if email not verified

### Updated Login Response (Recommended)
```json
{
    "success": true,
    "message": "Login successful.",
    "data": {
        "token": "...",
        "user": {
            "id": 1,
            "name": "Ganesh",
            "email": "ganesh@yopmail.com",
            "email_verified": false,  // ← Add this
            "status": "active",
            "timezone": "UTC"
        }
    }
}
```

---

## Summary

| Field | Purpose | Registration Value | Login Check |
|-------|---------|-------------------|-------------|
| `status` | Account state | `'active'` (default) | ✅ **Required**: Must be `'active'` |
| `email_verified_at` | Email verification | `null` | ⚠️ **Optional**: Currently not checked |

**Key Points:**
- ✅ `status = 'active'` is **required** for login (already implemented)
- ✅ `email_verified_at = null` on registration is **correct** (user hasn't verified yet)
- ⚠️ Email verification check is **optional** (currently commented out)
- 💡 You can uncomment the email verification check if you want to require it

