# Solution: Distinguishing Draft vs Publish Actions

## 🔍 Problem Analysis

You identified that when clicking both "Save Draft" and "Publish" buttons, the payload was the same, making it impossible for the backend to distinguish between the two actions.

## ✅ Solution Implemented

### 1. Added `status` Field to Payload

**Fixed in:** `lib/postsService.ts`

The frontend now **always includes** a `status` field in the payload when saving posts:

-   **Save Draft**: Sends `status: "draft"` in the payload
-   **Publish**:
    -   First saves with `status: "draft"` (if not already saved)
    -   Then calls `/api/posts/{id}/publish` endpoint to change status to `"posted"`

### 2. How Backend Can Identify the Action

#### For Create/Update Endpoints (`POST /api/posts` or `PUT /api/posts/{id}`):

```php
// In your Laravel controller
$validated = $request->validate([
    'status' => 'required|in:draft,posted', // This field is now always present
    // ... other fields
]);

// Check the status field
if ($validated['status'] === 'draft') {
    // User clicked "Save Draft"
    $post->status = 'draft';
} elseif ($validated['status'] === 'posted') {
    // User wants to publish directly (optional feature)
    $post->status = 'posted';
    $post->published_at = now();
}
```

#### For Publish Endpoint (`PUT /api/posts/{id}/publish`):

```php
// This endpoint is ONLY called when user clicks "Publish"
public function publish(Request $request, $id) {
    $post = Post::findOrFail($id);

    // Change status from 'draft' to 'posted'
    $post->status = 'posted';
    $post->published_at = now();
    $post->save();

    return response()->json([
        'success' => true,
        'message' => 'Post published successfully',
        'data' => $post,
    ]);
}
```

## 📋 Current Flow

### When User Clicks "Save Draft":

```
User clicks "Save Draft"
    ↓
Frontend: handleSaveDraft()
    ↓
Frontend: saveDraft() → POST /api/posts or PUT /api/posts/{id}
    ↓
Payload includes: { status: "draft", content: "...", ... }
    ↓
Backend: Receives status = "draft"
    ↓
Backend: Saves post with status = "draft"
```

### When User Clicks "Publish":

```
User clicks "Publish"
    ↓
Frontend: handlePublish()
    ↓
Step 1: saveDraft() → POST /api/posts or PUT /api/posts/{id}
    ↓
Payload includes: { status: "draft", content: "...", ... }
    ↓
Backend: Saves post with status = "draft"
    ↓
Step 2: publishPost() → PUT /api/posts/{id}/publish
    ↓
Backend: Changes status from "draft" to "posted"
    ↓
Backend: Sets published_at = now()
```

## 🔑 Key Points

1. **`status` field is now always included** in create/update requests
2. **Backend can check `status` field** to identify the action:
    - `status === "draft"` → Save Draft action
    - `status === "posted"` → Publish action (optional, not currently used)
3. **Publish endpoint** (`/publish`) is ONLY called for publish action
4. **Two-step process** for publish: Save as draft first, then publish

## 📝 Changes Made

### File: `lib/postsService.ts`

**Added `status: "draft"` to:**

-   `createPost()` function (both FormData and JSON)
-   `updatePost()` function (both FormData and JSON)

**Example:**

```typescript
// Before
body = JSON.stringify({
    content: postData.content,
    category: postData.category || "General",
    platforms: postData.platforms || [],
});

// After
body = JSON.stringify({
    content: postData.content,
    status: "draft", // ✅ Added this field
    category: postData.category || "General",
    platforms: postData.platforms || [],
});
```

## 📚 Documentation Created

1. **`LARAVEL_API_IMPLEMENTATION_GUIDE.md`** - Complete API documentation with:
    - All endpoints with request/response examples
    - Laravel controller implementation code
    - Database schema
    - Validation rules
    - How to identify draft vs publish

## 🧪 Testing

To test the solution:

1. **Test Save Draft:**

    - Click "Save Draft" button
    - Check network tab - payload should include `status: "draft"`
    - Backend should receive and save with `status = "draft"`

2. **Test Publish:**
    - Click "Publish" button
    - Check network tab - first request should include `status: "draft"`
    - Second request should be to `/publish` endpoint
    - Backend should change status to `"posted"`

## 🚀 Next Steps for Laravel Implementation

1. **Update your Laravel controller** to check the `status` field:

    ```php
    $validated = $request->validate([
        'status' => 'required|in:draft,posted',
        // ... other fields
    ]);
    ```

2. **In the `store()` method**, use the status field:

    ```php
    $post = Post::create([
        'status' => $validated['status'],
        'published_at' => $validated['status'] === 'posted' ? now() : null,
        // ... other fields
    ]);
    ```

3. **In the `update()` method**, check if post is draft before allowing update:

    ```php
    if ($post->status !== 'draft') {
        return response()->json([
            'error' => 'Only draft posts can be updated'
        ], 400);
    }
    ```

4. **Implement the `publish()` method** to change status:
    ```php
    public function publish(Request $request, $id) {
        $post = Post::findOrFail($id);
        $post->status = 'posted';
        $post->published_at = now();
        $post->save();
        // ... return response
    }
    ```

## ✅ Summary

-   ✅ **Problem Solved**: Backend can now distinguish between draft and publish actions
-   ✅ **Solution**: Added `status` field to all create/update requests
-   ✅ **Documentation**: Complete Laravel API guide created
-   ✅ **Code Updated**: `postsService.ts` now includes status field

The backend can now identify:

-   **Save Draft** → `status: "draft"` in payload
-   **Publish** → `/publish` endpoint is called

---

**Date:** 2024-01-15
**Status:** ✅ Complete
