# AppUser Separation Guide

## Overview
The application now has **two separate user models**:
- **`User`** - For admin users (admin panel)
- **`AppUser`** - For frontend/API users (Next.js frontend)

## Models

### User Model (`app/Models/User.php`)
- **Purpose**: Admin users only
- **Table**: `users`
- **Features**: 
  - Role-based access (admin/subscriber)
  - Subscription management
  - **No API tokens** (removed `HasApiTokens` trait)
- **Used by**: Admin panel authentication

### AppUser Model (`app/Models/AppUser.php`)
- **Purpose**: Frontend/API users
- **Table**: `app_users`
- **Features**:
  - API token authentication (has `HasApiTokens` trait)
  - Status management (active/inactive/suspended)
  - Timezone support
  - **No role field** (simpler structure)
- **Used by**: Next.js frontend API

## Database Structure

### `users` Table (Admin)
```sql
- id
- name
- email
- password
- role (admin/subscriber)
- timezone
- status
- email_verified_at
- remember_token
- timestamps
```

### `app_users` Table (Frontend)
```sql
- id
- name
- email
- password
- timezone (default: UTC)
- status (active/inactive/suspended)
- email_verified_at
- remember_token
- timestamps
```

## API Authentication

### Login Endpoint
```
POST /api/login
```

**Uses**: `AppUser` model
**Returns**: Token for API authentication

### Protected Routes
```
GET /api/user
POST /api/logout
GET /api/posts
...
```

**Uses**: Sanctum token authentication
**Token Model**: `AppUser` (stored in `personal_access_tokens` table)

## Migration

Run the migration to create the `app_users` table:

```bash
php artisan migrate
```

This will create the `app_users` table with the appropriate structure.

## Configuration

### Auth Config (`config/auth.php`)
Added `app_users` provider:
```php
'app_users' => [
    'driver' => 'eloquent',
    'model' => App\Models\AppUser::class,
],
```

### Sanctum Configuration
Sanctum automatically works with `AppUser` because:
- Tokens are stored in `personal_access_tokens` table
- `tokenable_type` column stores the model class (`App\Models\AppUser`)
- `tokenable_id` stores the user ID

## Usage Examples

### Register New App User
```php
$appUser = AppUser::create([
    'name' => 'John Doe',
    'email' => 'john@example.com',
    'password' => Hash::make('password123'),
]);
```

### Login (in AuthController)
```php
$user = AppUser::where('email', $email)->first();
if ($user && Hash::check($password, $user->password)) {
    $token = $user->createToken('web_app_token')->plainTextToken;
}
```

### Get Authenticated User (in Protected Routes)
```php
// In controller with auth:sanctum middleware
$appUser = $request->user(); // Returns AppUser instance
```

## Benefits of Separation

1. **Security**: Admin and frontend users are completely separate
2. **Simplicity**: AppUser has simpler structure (no roles)
3. **Scalability**: Can add different features to each user type
4. **Maintenance**: Easier to manage and update separately
5. **Performance**: Smaller table for app users

## Important Notes

- **Admin users** (`User` model) cannot use API tokens
- **App users** (`AppUser` model) cannot access admin panel
- Tokens created by `AppUser` are stored in `personal_access_tokens` with `tokenable_type = 'App\Models\AppUser'`
- The `auth:sanctum` middleware automatically resolves the correct model from the token

## Testing

### Create Test App User
```php
AppUser::create([
    'name' => 'Test User',
    'email' => 'test@example.com',
    'password' => Hash::make('password123'),
    'status' => 'active',
]);
```

### Test Login
```bash
curl -X POST http://localhost:8000/api/login \
  -H "Content-Type: application/json" \
  -d '{"email":"test@example.com","password":"password123"}'
```

