# API Quick Reference Guide

## API Endpoints

### Authentication Endpoints

#### Login
```
POST /api/login
Content-Type: application/json

Body:
{
  "email": "user@example.com",
  "password": "password123"
}

Response (Success):
{
  "success": true,
  "message": "Login successful.",
  "data": {
    "token": "1|abcdef1234567890...",
    "user": {
      "id": 1,
      "name": "John Doe",
      "email": "user@example.com",
      "role": "user",
      "timezone": "UTC",
      "status": "active"
    }
  }
}
```

#### Logout
```
POST /api/logout
Authorization: Bearer {token}

Response:
{
  "success": true,
  "message": "Logged out successfully."
}
```

#### Get Current User
```
GET /api/user
Authorization: Bearer {token}

Response:
{
  "success": true,
  "data": {
    "id": 1,
    "name": "John Doe",
    "email": "user@example.com",
    ...
  }
}
```

## How to Call API from Next.js

### 1. Setup API Client

```typescript
// lib/api.ts
import axios from 'axios';

const apiClient = axios.create({
  baseURL: process.env.NEXT_PUBLIC_API_URL + '/api',
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
});

// Add token to all requests
apiClient.interceptors.request.use((config) => {
  const token = localStorage.getItem('auth_token');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

export default apiClient;
```

### 2. Login Example

```typescript
const response = await apiClient.post('/login', {
  email: 'user@example.com',
  password: 'password123'
});

// Store token
localStorage.setItem('auth_token', response.data.data.token);
```

### 3. Protected Request Example

```typescript
// Token is automatically added by interceptor
const response = await apiClient.get('/user');
console.log(response.data);
```

## Token Storage

### ✅ Recommended: localStorage

```typescript
// Store
localStorage.setItem('auth_token', token);

// Retrieve
const token = localStorage.getItem('auth_token');

// Remove
localStorage.removeItem('auth_token');
```

**Why localStorage?**
- Persists across browser sessions
- Easy to access in Next.js client components
- Standard for SPAs
- Works well with axios interceptors

## How Sanctum Tokens Work

### Token Creation (Login)
1. User logs in with email/password
2. Laravel validates credentials
3. Laravel creates token in `personal_access_tokens` table
4. Token is hashed before storage
5. Plain text token returned once: `1|abcdef123456...`

### Token Validation (Protected Routes)
1. Frontend sends token in `Authorization: Bearer {token}` header
2. Laravel extracts token
3. Looks up token in database
4. Compares hashed versions
5. If valid, loads user and allows access

### Token Revocation (Logout)
1. Frontend calls `/api/logout`
2. Laravel deletes token from database
3. Token becomes invalid immediately

## API Route Configuration Explained

### Route File: `routes/api.php`

```php
// Public route - no authentication
Route::post('/login', [AuthController::class, 'login']);
// Creates: POST http://localhost:8000/api/login

// Protected route - requires token
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/user', [AuthController::class, 'profile']);
    // Creates: GET http://localhost:8000/api/user
    // Requires: Authorization: Bearer {token}
});
```

### How Routes Map to URLs

| Route Definition | Full URL | Method | Auth Required |
|----------------|----------|--------|---------------|
| `Route::post('/login', ...)` | `/api/login` | POST | No |
| `Route::get('/user', ...)` | `/api/user` | GET | Yes |
| `Route::post('/logout', ...)` | `/api/logout` | POST | Yes |

**Note:** All routes in `api.php` are automatically prefixed with `/api`

## Complete Next.js Integration

See `NEXTJS_API_INTEGRATION_GUIDE.md` for:
- Complete setup instructions
- Full code examples
- Authentication flow
- Error handling
- Protected routes
- CORS configuration

## Quick Start Checklist

1. ✅ API routes configured in `routes/api.php`
2. ✅ Login endpoint: `POST /api/login`
3. ✅ Logout endpoint: `POST /api/logout`
4. ✅ Protected routes use `auth:sanctum` middleware
5. ✅ Next.js: Install axios
6. ✅ Next.js: Create API client with interceptors
7. ✅ Next.js: Store token in localStorage
8. ✅ Next.js: Add token to Authorization header
9. ✅ Configure CORS in Laravel if needed

## Environment Variables

### Laravel (.env)
```env
APP_URL=http://localhost:8000
```

### Next.js (.env.local)
```env
NEXT_PUBLIC_API_URL=http://localhost:8000
```

## Testing

### Test Login
```bash
curl -X POST http://localhost:8000/api/login \
  -H "Content-Type: application/json" \
  -d '{"email":"user@example.com","password":"password123"}'
```

### Test Protected Route
```bash
curl -X GET http://localhost:8000/api/user \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

