# API Exception Handling Fix

## Problem
When accessing protected API routes (like `/api/user`), you were getting:
```
Route [login] not defined
```

This happened because Laravel was trying to redirect unauthenticated API requests to a 'login' route that doesn't exist.

## Solution
Updated `bootstrap/app.php` to handle API exceptions properly by returning JSON responses instead of redirects.

## What Was Fixed

### Before
- API authentication failures tried to redirect to `route('login')`
- This caused "Route [login] not defined" error
- API routes should return JSON, not redirects

### After
- API authentication failures return JSON: `{"success": false, "message": "Unauthenticated."}`
- API route not found errors return JSON: `{"success": false, "message": "Route not found."}`
- Proper HTTP status codes (401, 404)

## Exception Handlers Added

### 1. Authentication Exception Handler
```php
$exceptions->render(function (\Illuminate\Auth\AuthenticationException $e, $request) {
    if ($request->is('api/*')) {
        return response()->json([
            'success' => false,
            'message' => 'Unauthenticated.',
        ], 401);
    }
});
```

**When it triggers:**
- Token is missing
- Token is invalid
- Token has expired
- User is not authenticated

**Response:**
```json
{
    "success": false,
    "message": "Unauthenticated."
}
```
**Status Code:** 401

### 2. Route Not Found Exception Handler
```php
$exceptions->render(function (\Symfony\Component\Routing\Exception\RouteNotFoundException $e, $request) {
    if ($request->is('api/*')) {
        return response()->json([
            'success' => false,
            'message' => 'Route not found.',
        ], 404);
    }
});
```

**When it triggers:**
- API route doesn't exist
- Route name doesn't exist (like `route('login')`)

**Response:**
```json
{
    "success": false,
    "message": "Route not found."
}
```
**Status Code:** 404

## Testing

### Test 1: Valid Token
```http
GET http://localhost:8000/api/user
Authorization: Bearer {valid_token}
Accept: application/json
```

**Expected:** 200 OK with user data

### Test 2: Missing Token
```http
GET http://localhost:8000/api/user
Accept: application/json
```

**Expected:** 401 Unauthorized
```json
{
    "success": false,
    "message": "Unauthenticated."
}
```

### Test 3: Invalid Token
```http
GET http://localhost:8000/api/user
Authorization: Bearer invalid_token_here
Accept: application/json
```

**Expected:** 401 Unauthorized
```json
{
    "success": false,
    "message": "Unauthenticated."
}
```

### Test 4: Non-existent Route
```http
GET http://localhost:8000/api/nonexistent
Authorization: Bearer {token}
Accept: application/json
```

**Expected:** 404 Not Found
```json
{
    "success": false,
    "message": "Route not found."
}
```

## Common Issues Fixed

### Issue 1: "Route [login] not defined"
✅ **Fixed**: API routes now return JSON instead of trying to redirect

### Issue 2: HTML Error Pages for API
✅ **Fixed**: All API errors return JSON responses

### Issue 3: Wrong Status Codes
✅ **Fixed**: Proper HTTP status codes (401, 404)

## How It Works

1. **Request comes to API route** (`/api/user`)
2. **Sanctum middleware** checks authentication
3. **If unauthenticated**: Throws `AuthenticationException`
4. **Exception handler** catches it
5. **Checks if API route**: `$request->is('api/*')`
6. **Returns JSON** instead of redirecting

## Important Notes

- ✅ Only affects API routes (`/api/*`)
- ✅ Web routes still work normally (can redirect)
- ✅ Proper JSON responses for all API errors
- ✅ Consistent error format across all API endpoints

## Summary

The fix ensures that:
1. API authentication failures return JSON (not redirects)
2. API route errors return JSON (not HTML)
3. Proper HTTP status codes are used
4. Consistent error response format

Now your API will work correctly with proper error handling!

